from typing import Annotated

from fastapi import APIRouter, Depends, Query, status
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user, authorize
from ..databases import get_async_session
from . import enums, schemas, utils
from .permissions import permission_mapper


router = APIRouter(tags=["tags"], prefix="/tag")


@router.post(
    "/",
    response_model=schemas.Tag,
    status_code=status.HTTP_201_CREATED,
)
async def create_tag(
    tag: schemas.TagCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_tag(session, tag, request_user)


@router.get(
    "/",
    response_model=schemas.PaginatedTag,
)
async def list_tag(
    pagination: PaginationQuery = Depends(),
    filters: schemas.TagFilter = Depends(),
    ordering: Annotated[list[enums.TagOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_tags(session, pagination, filters, ordering, request_user)


@router.get(
    "/options/",
    response_model=schemas.PaginatedTagOpt,
)
async def list_tag_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.TagFilter = Depends(),
    ordering: Annotated[list[enums.TagOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_tags(session, pagination, filters, ordering, request_user)


@router.put(
    "/{tag_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["update_tag"]))],
    response_model=schemas.Tag,
)
async def update_tag(
    tag: schemas.TagUpdate,
    tag_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_tag(session, tag_id, tag, request_user)


@router.delete(
    "/{tag_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["delete_tag"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_tag(
    tag_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_tag(session, tag_id, request_user)
