from decimal import Decimal
from typing_extensions import Annotated

from pydantic import PositiveInt, StringConstraints

from be_kit.paginations import BasePaginatedResponse
from be_kit.schemas import (
    BaseSchema,
    BaseORMSchema,
)


# Country


class Country(BaseORMSchema):
    name: Annotated[str, StringConstraints(max_length=100)]
    iso_a2: Annotated[str, StringConstraints(max_length=2)]
    iso_a3: Annotated[str, StringConstraints(max_length=3)]
    iso_n: PositiveInt


class PaginatedCountry(BasePaginatedResponse):
    items: list[Country]


class CountryFilter(BaseSchema):
    name: Annotated[str, StringConstraints(max_length=100)] | None = None
    iso_a2: Annotated[str, StringConstraints(max_length=2)] | None = None
    iso_a3: Annotated[str, StringConstraints(max_length=3)] | None = None
    iso_n: PositiveInt | None = None
    name__icontains: Annotated[str, StringConstraints(max_length=100)] | None = None
    iso_a2__icontains: Annotated[str, StringConstraints(max_length=2)] | None = None
    iso_a3__icontains: Annotated[str, StringConstraints(max_length=3)] | None = None


class CountryMin(Country):
    pass


class CountryOpt(CountryMin):
    pass


class PaginatedCountryOpt(BasePaginatedResponse):
    items: list[CountryOpt]


class CountryIdMixin:
    country_id: PositiveInt


# Currency


class BaseCurrency:
    name: Annotated[str, StringConstraints(max_length=100)]
    iso: Annotated[str, StringConstraints(max_length=3)]
    symbol: Annotated[str, StringConstraints(max_length=5)] | None


class Currency(
    BaseCurrency,
    BaseORMSchema,
):
    country: "Country | None"
    name: Annotated[str, StringConstraints(max_length=100)]
    iso: Annotated[str, StringConstraints(max_length=3)]
    symbol: Annotated[str, StringConstraints(max_length=5)] | None


class PaginatedCurrency(BasePaginatedResponse):
    items: list[Currency]


class CurrencyFilter(BaseSchema):
    country_id: PositiveInt | None = None
    name: Annotated[str, StringConstraints(max_length=100)] | None = None
    iso: Annotated[str, StringConstraints(max_length=3)] | None = None
    symbol: Annotated[str, StringConstraints(max_length=5)] | None = None
    country___name__icontains: (
        Annotated[str, StringConstraints(max_length=100)] | None
    ) = None
    name__icontains: Annotated[str, StringConstraints(max_length=100)] | None = None


class CurrencyMin(BaseCurrency, BaseORMSchema):
    pass


class CurrencyOpt(CurrencyMin):
    pass


class PaginatedCurrencyOpt(BasePaginatedResponse):
    items: list[CurrencyOpt]


# PhoneCode


class PhoneCode(
    BaseORMSchema,
):
    country: "Country | None"
    phone_code: Annotated[str, StringConstraints(max_length=10)]


class PaginatedPhoneCode(BasePaginatedResponse):
    items: list[PhoneCode]


class PhoneCodeFilter(BaseSchema):
    country_id: PositiveInt | None = None
    phone_code: Annotated[str, StringConstraints(max_length=10)] | None = None
    country___name__icontains: (
        Annotated[str, StringConstraints(max_length=100)] | None
    ) = None
    name__icontains: Annotated[str, StringConstraints(max_length=100)] | None = None


class PhoneCodeMin(PhoneCode):
    pass


class PhoneCodeOpt(PhoneCodeMin):
    pass


class PaginatedPhoneCodeOpt(BasePaginatedResponse):
    items: list[PhoneCodeOpt]


# VAT


class VAT(
    BaseORMSchema,
):
    country: "Country | None"
    name: Annotated[str, StringConstraints(max_length=50)]
    rate: Decimal


class PaginatedVAT(BasePaginatedResponse):
    items: list[VAT]


class VATFilter(BaseSchema):
    country_id: PositiveInt | None = None
    name: Annotated[str, StringConstraints(max_length=50)] | None = None
    country___name__icontains: (
        Annotated[str, StringConstraints(max_length=100)] | None
    ) = None
    name__icontains: Annotated[str, StringConstraints(max_length=100)] | None = None


class VATMin(VAT):
    pass


class VATOpt(VATMin):
    pass


class PaginatedVATOpt(BasePaginatedResponse):
    items: list[VATOpt]


# Bank


class Bank(BaseORMSchema):
    name: Annotated[str, StringConstraints(max_length=100)]
    legal_name: Annotated[str, StringConstraints(max_length=100)]
    code: Annotated[str, StringConstraints(max_length=20)]
    swift_code: Annotated[str, StringConstraints(max_length=11)] | None
    country: Country | None


class PaginatedBank(BasePaginatedResponse):
    items: list[Bank]


class BankFilter(BaseSchema):
    country_id: PositiveInt | None = None
    name: Annotated[str, StringConstraints(max_length=100)] | None = None
    legal_name: Annotated[str, StringConstraints(max_length=100)] | None = None
    code: Annotated[str, StringConstraints(max_length=20)] | None = None
    name__icontains: Annotated[str, StringConstraints(max_length=100)] | None = None
    legal_name__icontains: Annotated[str, StringConstraints(max_length=100)] | None = (
        None
    )


class BankMin(Bank):
    pass


class BankOpt(BankMin):
    pass


class PaginatedBankOpt(BasePaginatedResponse):
    items: list[BankOpt]
