from decimal import Decimal
from sqlalchemy import (
    BigInteger,
    ForeignKey,
    Numeric,
    String,
    UniqueConstraint,
)
from sqlalchemy.orm import Mapped, mapped_column, relationship

from be_kit.databases import BaseModel


class Country(BaseModel):
    __tablename__ = "core_services_country"

    name: Mapped[str] = mapped_column(String(100), unique=True)
    iso_a2: Mapped[str] = mapped_column(String(2), unique=True)
    iso_a3: Mapped[str] = mapped_column(String(3), unique=True)
    iso_n: Mapped[int] = mapped_column(BigInteger, unique=True)


class Currency(BaseModel):
    __tablename__ = "core_services_currency"

    country_id: Mapped[int | None] = mapped_column(
        BigInteger,
        ForeignKey(f"{Country.__tablename__}.pk", ondelete="CASCADE"),
        nullable=True,
    )
    country: Mapped["Country | None"] = relationship(
        Country,
        passive_deletes=True,
        lazy="selectin",
    )
    name: Mapped[str] = mapped_column(String(100), unique=True)
    iso: Mapped[str] = mapped_column(String(3), unique=True)
    symbol: Mapped[str] = mapped_column(String(5), nullable=True, unique=True)


class PhoneCode(BaseModel):
    __tablename__ = "core_services_phone_code"

    country_id: Mapped[int | None] = mapped_column(
        BigInteger,
        ForeignKey(f"{Country.__tablename__}.pk", ondelete="CASCADE"),
        nullable=True,
    )
    country: Mapped["Country | None"] = relationship(
        Country,
        passive_deletes=True,
        lazy="selectin",
    )
    phone_code: Mapped[str] = mapped_column(String(10), unique=True)


class VAT(BaseModel):
    __tablename__ = "core_services_vat"
    __table_args__ = (
        UniqueConstraint(
            "country_id", "name", name=f"uq_{__tablename__}_country_id_name"
        ),
    )

    country_id: Mapped[int | None] = mapped_column(
        BigInteger,
        ForeignKey(f"{Country.__tablename__}.pk", ondelete="CASCADE"),
        nullable=True,
    )
    country: Mapped["Country | None"] = relationship(
        Country,
        passive_deletes=True,
        lazy="selectin",
    )
    name: Mapped[str] = mapped_column(String(50))
    rate: Mapped[Decimal] = mapped_column(Numeric(5, 4))


class Bank(BaseModel):
    __tablename__ = "core_services_bank"
    __table_args__ = (
        UniqueConstraint(
            "code", "country_id", name=f"uq_{__tablename__}_code_country_id"
        ),
    )

    name: Mapped[str] = mapped_column(String(100))
    legal_name: Mapped[str] = mapped_column(String(100))
    code: Mapped[str] = mapped_column(String(20))
    swift_code: Mapped[str] = mapped_column(String(11))
    country_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey(f"{Country.__tablename__}.pk", ondelete="CASCADE"),
        nullable=False,
    )
    country: Mapped["Country"] = relationship(
        Country,
        passive_deletes=True,
        lazy="selectin",
    )
