from typing import Annotated

from fastapi import APIRouter, Depends, Query
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user
from ..databases import get_async_session
from . import enums, schemas, utils


# Country

country_router = APIRouter(tags=["countries"], prefix="/country")


@country_router.get(
    "/{country_id:int}/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.Country,
)
async def retrieve_country(
    country_id: int,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_country(session, country_id)


@country_router.get(
    "/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.PaginatedCountry,
)
async def list_country(
    pagination: PaginationQuery = Depends(),
    filters: schemas.CountryFilter = Depends(),
    ordering: Annotated[list[enums.CountryOrdering] | None, Query()] = None,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_countries(session, pagination, filters, ordering)


@country_router.get(
    "/options/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.PaginatedCountryOpt,
)
async def list_country_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.CountryFilter = Depends(),
    ordering: Annotated[list[enums.CountryOrdering] | None, Query()] = None,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_countries(session, pagination, filters, ordering)


# Currency

currency_router = APIRouter(tags=["currencies"], prefix="/currency")


@currency_router.get(
    "/{currency_id:int}/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.Currency,
)
async def retrieve_currency(
    currency_id: int,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_currency(session, currency_id)


@currency_router.get(
    "/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.PaginatedCurrency,
)
async def list_currency(
    pagination: PaginationQuery = Depends(),
    filters: schemas.CurrencyFilter = Depends(),
    ordering: Annotated[list[enums.CurrencyOrdering] | None, Query()] = None,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_currencies(session, pagination, filters, ordering)


@currency_router.get(
    "/options/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.PaginatedCurrencyOpt,
)
async def list_currency_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.CurrencyFilter = Depends(),
    ordering: Annotated[list[enums.CurrencyOrdering] | None, Query()] = None,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_currencies(session, pagination, filters, ordering)


# PhoneCode

phonecode_router = APIRouter(tags=["phonecodes"], prefix="/phonecode")


@phonecode_router.get(
    "/{phonecode_id:int}/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.PhoneCode,
)
async def retrieve_phonecode(
    phonecode_id: int,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_phonecode(session, phonecode_id)


@phonecode_router.get(
    "/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.PaginatedPhoneCode,
)
async def list_phonecode(
    pagination: PaginationQuery = Depends(),
    filters: schemas.PhoneCodeFilter = Depends(),
    ordering: Annotated[list[enums.PhoneCodeOrdering] | None, Query()] = None,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_phonecodes(session, pagination, filters, ordering)


@phonecode_router.get(
    "/options/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.PaginatedPhoneCodeOpt,
)
async def list_phonecode_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.PhoneCodeFilter = Depends(),
    ordering: Annotated[list[enums.PhoneCodeOrdering] | None, Query()] = None,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_phonecodes(session, pagination, filters, ordering)


# VAT

vat_router = APIRouter(tags=["vats"], prefix="/vat")


@vat_router.get(
    "/{vat_id:int}/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.VAT,
)
async def retrieve_vat(
    vat_id: int,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_vat(session, vat_id)


@vat_router.get(
    "/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.PaginatedVAT,
)
async def list_vat(
    pagination: PaginationQuery = Depends(),
    filters: schemas.VATFilter = Depends(),
    ordering: Annotated[list[enums.VATOrdering] | None, Query()] = None,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_vats(session, pagination, filters, ordering)


@vat_router.get(
    "/options/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.PaginatedVATOpt,
)
async def list_vat_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.VATFilter = Depends(),
    ordering: Annotated[list[enums.VATOrdering] | None, Query()] = None,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_vats(session, pagination, filters, ordering)


# Bank

bank_router = APIRouter(tags=["banks"], prefix="/bank")


@bank_router.get(
    "/{bank_id:int}/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.Bank,
)
async def retrieve_bank(
    bank_id: int,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_bank(session, bank_id)


@bank_router.get(
    "/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.PaginatedBank,
)
async def list_bank(
    pagination: PaginationQuery = Depends(),
    filters: schemas.BankFilter = Depends(),
    ordering: Annotated[list[enums.BankOrdering] | None, Query()] = None,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_banks(session, pagination, filters, ordering)


@bank_router.get(
    "/options/",
    dependencies=[Depends(get_request_user)],
    response_model=schemas.PaginatedBankOpt,
)
async def list_bank_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.BankFilter = Depends(),
    ordering: Annotated[list[enums.BankOrdering] | None, Query()] = None,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_banks(session, pagination, filters, ordering)
