from typing_extensions import Annotated

from fastapi import File, UploadFile
from pydantic import EmailStr, PositiveInt, StringConstraints
from pydantic_extra_types.phone_numbers import PhoneNumber

from be_kit.paginations import BasePaginatedResponse
from be_kit.schemas import BaseSchema, BaseORMSchema
from be_uam.kit.schemas import MetadataMixin
from be_uam.user.schemas import UserMin
from ..tag.schemas import TagListMixin, TagListWriteMixin
from . import enums


# Industry


class BaseIndustry:
    industry_type: enums.ContactIndustryType


class IndustryCreate(BaseIndustry, BaseSchema):
    pass


class IndustryUpdate(BaseIndustry, BaseSchema):
    pk: PositiveInt | None


class Industry(BaseIndustry, BaseORMSchema):
    pass


class IndustryValidator(BaseIndustry, BaseSchema):
    contact_name: Annotated[str, StringConstraints(min_length=1, max_length=100)]


# Address


class BaseAddress:
    address_type: enums.ContactAddressType
    name: Annotated[str, StringConstraints(min_length=1, max_length=100)]
    email: "EmailStr | None"
    address: Annotated[str, StringConstraints(min_length=1, max_length=255)]
    phone: PhoneNumber | None


class AddressCreate(BaseAddress, BaseSchema):
    pass


class AddressUpdate(BaseAddress, BaseSchema):
    pk: PositiveInt | None


class Address(BaseAddress, BaseORMSchema):
    pass


class AddressValidator(BaseAddress, BaseSchema):
    contact_name: Annotated[str, StringConstraints(min_length=1, max_length=100)]


# Contact


class BaseContact:
    name: Annotated[str, StringConstraints(min_length=1, max_length=100)]
    legal_type: enums.ContactLegalType
    tax_id: Annotated[str | None, StringConstraints(min_length=1, max_length=50)]


class ContactCreate(TagListWriteMixin, BaseContact, BaseSchema):
    industries: list[IndustryCreate]
    addresses: list[AddressCreate]


class ContactUpdate(TagListWriteMixin, BaseContact, BaseSchema):
    industries: list[IndustryUpdate]
    addresses: list[AddressUpdate]


class Contact(TagListMixin, MetadataMixin, BaseContact, BaseORMSchema):
    industries: list[Industry]
    addresses: list[Address]


class PaginatedContact(BasePaginatedResponse):
    items: list[Contact]


class ContactFilter(BaseSchema):
    name: str | None = None
    name__icontains: str | None = None
    legal_type: enums.ContactLegalType | None = None
    tax_id: str | None = None
    tax_id__icontains: str | None = None


class ContactMin(BaseContact, BaseORMSchema):
    pass


class ContactOpt(ContactMin):
    pass


class PaginatedContactOpt(BasePaginatedResponse):
    items: list[ContactOpt]


class ContactUpload(BaseSchema):
    file: Annotated[UploadFile, File()]


class ContactValidator(BaseContact, BaseSchema):
    pass


UserMin.model_rebuild()
