from sqlalchemy import (
    BigInteger,
    Enum,
    ForeignKey,
    String,
    UniqueConstraint,
)
from sqlalchemy.orm import Mapped, declared_attr, mapped_column, relationship

from be_kit.databases import BaseModel, SoftDeleteMixin
from be_uam.kit.models import MetadataMixin, OrganizationMixin
from ..tag.models import Tag
from . import enums


class Contact(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "core_services_contact"
    __table_args__ = (UniqueConstraint("organization_id", "name"),)

    name: Mapped[str] = mapped_column(String(100))
    legal_type: Mapped[enums.ContactLegalType] = mapped_column(
        Enum(enums.ContactLegalType), index=True
    )
    tax_id: Mapped[str | None] = mapped_column(String(50), nullable=True)
    industries: Mapped[list["Industry"]] = relationship(
        "Industry",
        cascade="all, delete-orphan",
        passive_deletes=True,
        lazy="selectin",
    )
    addresses: Mapped[list["Address"]] = relationship(
        "Address",
        cascade="all, delete-orphan",
        passive_deletes=True,
        lazy="selectin",
    )
    tags: Mapped[list[Tag]] = relationship(
        secondary="core_services_contact_tag", lazy="selectin"
    )


class Industry(BaseModel):
    __tablename__ = "core_services_contact_industry"
    __table_args__ = (UniqueConstraint("contact_id", "industry_type"),)

    contact_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(f"{Contact.__tablename__}.pk", ondelete="CASCADE")
    )
    contact: Mapped["Contact"] = relationship(
        back_populates="industries", passive_deletes=True, lazy="selectin"
    )
    industry_type: Mapped[enums.ContactIndustryType] = mapped_column(
        Enum(enums.ContactIndustryType), index=True
    )


class Address(BaseModel):
    __tablename__ = "core_services_contact_address"
    __table_args__ = (UniqueConstraint("contact_id", "name"),)

    contact_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(f"{Contact.__tablename__}.pk", ondelete="CASCADE")
    )
    contact: Mapped[Contact] = relationship(
        back_populates="addresses", passive_deletes=True, lazy="selectin"
    )
    address_type: Mapped[enums.ContactAddressType] = mapped_column(
        Enum(enums.ContactAddressType), index=True
    )
    name: Mapped[str] = mapped_column(String(100))
    email: Mapped[str | None] = mapped_column(String(100), nullable=True)
    address: Mapped[str] = mapped_column(String(255))
    phone: Mapped[str | None] = mapped_column(String(50), nullable=True)


class ContactTag(BaseModel):
    __tablename__ = "core_services_contact_tag"
    __table_args__ = (UniqueConstraint("contact_id", "tag_id"),)

    contact_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey("core_services_contact.pk", ondelete="CASCADE"),
        index=True,
    )
    tag_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey("core_services_tag.pk", ondelete="CASCADE"),
        index=True,
    )
