import io
from typing import Annotated

from isort import file
import pandas as pd
from fastapi import APIRouter, Depends, Form, Query, status
from fastapi.responses import StreamingResponse
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_kit.exim.utils import get_version_upload_template
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user, authorize
from ..databases import get_async_session
from . import enums, schemas, utils
from .permissions import permission_mapper


router = APIRouter(tags=["contacts"], prefix="/contact")


@router.post(
    "/",
    dependencies=[Depends(authorize(permission_mapper["create_contact"]))],
    response_model=schemas.Contact,
    status_code=status.HTTP_201_CREATED,
)
async def create_contact(
    contact: schemas.ContactCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_contact(session, contact, request_user)


@router.get(
    "/{contact_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["retrieve_contact"]))],
    response_model=schemas.Contact,
)
async def retrieve_contact(
    contact_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_contact(session, contact_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(permission_mapper["list_contact"]))],
    response_model=schemas.PaginatedContact,
)
async def list_contact(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ContactFilter = Depends(),
    ordering: Annotated[list[enums.ContactOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_contacts(
        session, pagination, filters, ordering, request_user
    )


@router.get(
    "/options/",
    dependencies=[Depends(authorize(permission_mapper["list_contact_option"]))],
    response_model=schemas.PaginatedContactOpt,
)
async def list_contact_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ContactFilter = Depends(),
    ordering: Annotated[list[enums.ContactOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_contacts(
        session, pagination, filters, ordering, request_user
    )


@router.put(
    "/{contact_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["update_contact"]))],
    response_model=schemas.Contact,
)
async def update_contact(
    contact: schemas.ContactUpdate,
    contact_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_contact(session, contact_id, contact, request_user)


@router.delete(
    "/{contact_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["delete_contact"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_contact(
    contact_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_contact(session, contact_id, request_user)


@router.get(
    "/upload/template/",
    dependencies=[
        Depends(authorize(permission_mapper["download_upload_contact_template"]))
    ],
)
async def download_upload_contact_template():
    contact_template = await get_version_upload_template(schemas.ContactValidator)
    industry_template = await get_version_upload_template(schemas.IndustryValidator)
    address_template = await get_version_upload_template(schemas.AddressValidator)

    buffer = io.BytesIO()
    with pd.ExcelWriter(buffer, engine="openpyxl") as writer:
        contact_template.to_excel(writer, sheet_name="Contact", index=False)
        industry_template.to_excel(writer, sheet_name="Industry", index=False)
        address_template.to_excel(writer, sheet_name="Address", index=False)
    buffer.seek(0)

    return StreamingResponse(
        buffer,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": 'attachment; filename="Contact Template.xlsx"'},
    )


@router.post(
    "/upload/",
    dependencies=[Depends(authorize(permission_mapper["upload_contact"]))],
    status_code=status.HTTP_201_CREATED,
)
async def upload_contact(
    contact_upload: Annotated[
        schemas.ContactUpload, Form(media_type="multipart/form-data")
    ],
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.handle_contact_upload(session, contact_upload, request_user)


@router.get(
    "/download/",
    dependencies=[Depends(authorize(permission_mapper["download_contact"]))],
)
async def download_contact(
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    buffer = await utils.handle_contact_download(session, request_user)
    return StreamingResponse(
        buffer,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": 'attachment; filename="Contacts.xlsx"'},
    )
