# symbolix-core_services

Core Services

## Conda Setup

To set up your development environment, create a new Conda environment with Python 3.12.3:

```bash
conda create --prefix .venv/ python=3.12.3
```

After creating the environment, make sure to activate it before proceeding with the next steps:

```bash
conda activate .venv/
```

Next, create a copy of `pip.conf.template` and replace `{PERSONAL_ACCESS_TOKEN}` with your GitLab personal access token. Place the updated file at `.venv/pip.conf` to enable access to private repositories.

## Install/Upgrade Development Kit

To install all dependencies from the remote server, run:

```bash
pip install -e .[dev] --config-settings editable_mode=strict
```

If you need to make changes to the dependencies' source code, install them from local folders instead:

```bash
pip install -e ../symbolix-be-py_kit[all] --config-settings editable_mode=strict
pip install -e ../symbolix-erp-uam --config-settings editable_mode=strict
pip install -e . --config-settings editable_mode=strict
```

This ensures that any local changes to dependencies are reflected in your environment.

## Running Development Server

To start the development server, simply run:

```bash
be-core_services
```

This will launch the application using the current environment settings.

To start the worker, run:

```bash
be-core_services-worker
```

## Git Management

### Precommit

To ensure code quality and consistency, run the pre-commit hooks before pushing your changes. This will automatically check and format your code according to the project's standards:

```bash
be-precommit
```

If any issues are found, fix them and re-run the command until all checks pass.

### Commit Message

Follow the [Conventional Commits](https://www.conventionalcommits.org/en/v1.0.0/) specification for your commit messages. This standard helps automate versioning and changelog generation.

A conventional commit message consists of a type, an optional scope, and a concise description:

```
<type>[optional scope]: <description>
```

**Types include:**

-   `feat`: A new feature
-   `fix`: A bug fix
-   `docs`: Documentation changes
-   `style`: Code style changes (formatting, missing semi colons, etc.)
-   `refactor`: Code changes that neither fix a bug nor add a feature
-   `perf`: Performance improvements
-   `test`: Adding or updating tests
-   `chore`: Changes to the build process or auxiliary tools

**Examples:**

```
feat(auth): add JWT authentication
fix(database): resolve connection leak
docs(readme): update setup instructions
```

Refer to the [Conventional Commits documentation](https://www.conventionalcommits.org/en/v1.0.0/) for more details and examples.

### Tagging a Release

Release and dev package creation is managed by CI/CD:

-   **Development Package:**
    Pushing to the `develop` branch triggers CI to build and publish a dev package.
    Manually bump the version using `bump2version [major|minor|patch|build]` before pushing.

-   **Release Package:**
    Creating or pushing a tag in GitLab will trigger CI to build and publish a release package. Normally, tags are created manually using:

    ```bash
    bump2version --tag release
    git push && git push --tags
    ```

    Make sure to push both commits and tags to ensure the release process is triggered.

## Migrations

To manage database migrations, use Alembic:

-   Create a new migration:

    ```bash
    alembic revision --autogenerate -m "core_services: [migration message]"
    ```

-   Apply the latest migrations:

    ```bash
    alembic upgrade head
    ```

-   Revert the last migration:

    ```bash
    alembic downgrade -1
    ```

## Build and Run in Binary Mode

To build the application as a standalone binary using PyInstaller:

```bash
pyinstaller \
    --distpath bin/dist \
    --workpath bin/build \
    --clean bin/api.spec
```

After building, run the binary:

```bash
./bin/dist/api
```

The same approach can be applied for worker services:

```bash
pyinstaller \
    --distpath bin/dist \
    --workpath bin/build \
    --clean bin/worker.spec

./bin/dist/worker
```

## Manually Publish a Package Release

To manually publish a new package release, ensure the following environment variables are set with the appropriate values in your `.env` file:

```env
BE_CI_PACKAGE_REGISTRY=http://example.com
BE_CI_AUTH_USERNAME=string
BE_CI_AUTH_TOKEN=secret
```

These variables configure the package registry URL and authentication credentials required for publishing.

Next, build and push the package to the registry using:

```bash
be-publish
```

## Build Docker Image

To build a Docker image for the application, use:

```bash
export $(grep BE_CI_AUTH_TOKEN .env | xargs)

docker build \
    --rm --no-cache \
    --build-arg AUTH_TOKEN="$BE_CI_AUTH_TOKEN" \
    -f .docker/Dockerfile \
    -t symbolix-core_services:latest .
```

The same approach can be applied for worker services:

```bash
docker build \
    --rm --no-cache \
    --build-arg AUTH_TOKEN="$BE_CI_AUTH_TOKEN" \
    -f .docker/worker.Dockerfile \
    -t symbolix-core_services-worker:latest .
```

## Run Standalone Docker Image

To run the application in a Docker container:

```bash
docker run --name symbolix-core_services \
    -p 55502:8000 \
    --network=symbolix \
    --env-file .env.local \
    --restart=always \
    --detach \
    symbolix-core_services:latest
```

The same approach can be applied for worker services:

```bash
docker run --name symbolix-core_services-worker \
    --network=symbolix \
    --env-file .env.local \
    --restart=always \
    --detach \
    symbolix-core_services-worker:latest
```
