import functools
import asyncio
from uuid import uuid4

from celery import Celery
from celery.signals import after_setup_logger, after_setup_task_logger

from .logging import InterceptHandler, setup_logging
from .settings import RedisSettings, TaskSettings


loop = asyncio.new_event_loop()
asyncio.set_event_loop(loop)


def get_broker(broker_settings: RedisSettings):
    return f"redis://{broker_settings.host}:{broker_settings.port}/{broker_settings.database}"


class TaskHandler:
    def __init__(
        self,
        app: str,
        settings: TaskSettings,
        broker_settings: RedisSettings,
        debug: bool = False,
    ):
        self._app = app
        self._settings = settings
        self._broker_settings = broker_settings
        self._client = Celery(
            app,
            broker=get_broker(broker_settings),
            backend=get_broker(broker_settings),
            timezone="Asia/Jakarta",
        )
        self._debug = debug

    @property
    def app(self) -> str:
        return self._app

    @property
    def settings(self) -> TaskSettings:
        return self._settings

    @property
    def broker_settings(self) -> RedisSettings:
        return self._broker_settings

    @property
    def client(self) -> Celery:
        return self._client

    @property
    def debug(self) -> bool:
        return self._debug

    def get_task_id(self) -> str:
        return str(uuid4())

    def task(self, *args, **kwargs):
        def _decorator(func):
            queue_name = f"{self.app}.{func.__name__}"
            task_kwargs = {**kwargs, "queue": queue_name}

            @self._client.task(*args, **task_kwargs)
            @functools.wraps(func)
            def _decorated(*f_args, **f_kwargs):
                return func(*f_args, **f_kwargs)

            return _decorated

        return _decorator

    def async_task(self, *args, **kwargs):
        def _decorator(func):
            queue_name = f"{self.app}.{func.__name__}"
            task_kwargs = {**kwargs, "queue": queue_name}

            @self._client.task(*args, **task_kwargs)
            @functools.wraps(func)
            def _decorated(*f_args, **f_kwargs):
                return loop.run_until_complete(func(*f_args, **f_kwargs))

            return _decorated

        return _decorator

    def run_worker(self, queue_names: list[str]):
        setup_logging()

        queues = ",".join(queue_names)
        argv = [
            "worker",
            "--concurrency=1",
            f"--loglevel={'debug' if self.debug else 'info'}",
            "-Q",
            queues,
            "-n",
            f"{self.app}@%h",
        ]
        self._client.worker_main(argv)


@after_setup_logger.connect
def setup_celery_logger(*args, **kwargs):
    logger_ = kwargs.get("logger")
    if logger_:
        logger_.handlers.clear()
        logger_.addHandler(InterceptHandler())


@after_setup_task_logger.connect
def setup_celery_task_logger(*args, **kwargs):
    logger_ = kwargs.get("logger")
    if logger_:
        logger_.handlers.clear()
        logger_.addHandler(InterceptHandler())
