import asyncssh
from fastapi import UploadFile
from fastapi import HTTPException

from .settings import SFTPSettings


class AsyncSFTPClient:
    def __init__(self, settings: SFTPSettings):
        self._settings = settings

    @property
    def settings(self) -> SFTPSettings:
        return self._settings

    @settings.setter
    def settings(self, value: SFTPSettings):
        self._settings = value

    def _full_remote_path(self, remote_path: str) -> str:
        if remote_path.startswith("/"):
            remote_path = remote_path[1:]
        return f"{self.settings.base_path}/{remote_path}"

    async def aupload(
        self,
        file: UploadFile,
        remote_path: str,
        max_file_size: int,
        overwrite: bool = False,
    ):
        remote_path = self._full_remote_path(remote_path)
        async with asyncssh.connect(
            self.settings.host,
            port=self.settings.port,
            username=self.settings.username,
            password=self.settings.password,
            known_hosts=None,
        ) as conn:
            async with conn.start_sftp_client() as sftp:
                try:
                    await sftp.stat(remote_path)
                    if not overwrite:
                        raise HTTPException(
                            status_code=409, detail="File already exists"
                        )
                except asyncssh.sftp.SFTPNoSuchFile:
                    pass

                parent_dir = remote_path.rpartition("/")[0]
                if parent_dir:
                    try:
                        await sftp.stat(parent_dir)
                    except asyncssh.sftp.SFTPNoSuchFile:
                        await sftp.makedirs(parent_dir)
                file_size = 0
                async with sftp.open(remote_path, "wb") as remote_file:
                    while chunk := await file.read():
                        file_size += len(chunk)
                        if file_size > max_file_size:
                            raise HTTPException(
                                status_code=413, detail="File size exceeds limit"
                            )
                        await remote_file.write(chunk)

    async def adelete(self, remote_path: str):
        remote_path = self._full_remote_path(remote_path)
        async with asyncssh.connect(
            self.settings.host,
            port=self.settings.port,
            username=self.settings.username,
            password=self.settings.password,
            known_hosts=None,
        ) as conn:
            async with conn.start_sftp_client() as sftp:
                await sftp.remove(remote_path)

    async def aexists(self, remote_path: str) -> bool:
        remote_path = self._full_remote_path(remote_path)
        async with asyncssh.connect(
            self.settings.host,
            port=self.settings.port,
            username=self.settings.username,
            password=self.settings.password,
            known_hosts=None,
        ) as conn:
            async with conn.start_sftp_client() as sftp:
                try:
                    await sftp.stat(remote_path)
                    return True
                except FileNotFoundError:
                    return False

    async def adownload(self, remote_path: str, local_path: str):
        remote_path = self._full_remote_path(remote_path)
        async with asyncssh.connect(
            self.settings.host,
            port=self.settings.port,
            username=self.settings.username,
            password=self.settings.password,
            known_hosts=None,
        ) as conn:
            async with conn.start_sftp_client() as sftp:
                async with sftp.open(remote_path, "rb") as remote_file:
                    with open(local_path, "wb") as local_file:
                        while True:
                            chunk = await remote_file.read(8192)
                            if not chunk:
                                break
                            local_file.write(chunk)

    async def astream(self, remote_path: str):
        remote_path = self._full_remote_path(remote_path)
        try:
            async with asyncssh.connect(
                self.settings.host,
                port=self.settings.port,
                username=self.settings.username,
                password=self.settings.password,
                known_hosts=None,
            ) as conn:
                async with conn.start_sftp_client() as sftp:
                    async with sftp.open(remote_path, "rb") as remote_file:
                        while True:
                            chunk = await remote_file.read(8192)
                            if not chunk:
                                break
                            yield chunk
        except FileNotFoundError as exc:
            raise HTTPException(404, "File not found") from exc
        except Exception as exc:
            raise HTTPException(500, f"SFTP error: {str(exc)}") from exc
