from datetime import datetime
from typing import Annotated

from pydantic import BaseModel, ConfigDict, AfterValidator
from pydantic.types import condecimal

from . import enums
from decimal import Decimal
from pydantic import Field


def validate_zero_to_one(value: float | None) -> float | None:
    if value is not None and not (0 <= value <= 1):
        raise ValueError("Value must be between 0 and 1 (inclusive).")
    return value


class BaseSchema(BaseModel):
    pass


class BaseORMSchema(BaseModel):
    model_config = ConfigDict(from_attributes=True)

    pk: int


RatioType = Annotated[Decimal, Field(gt=-1, lt=1, max_digits=16, decimal_places=15)]
PercentageType = Annotated[
    Decimal,
    Field(
        gt=-999999.999999999999999,
        lt=999999.999999999999999,
        max_digits=21,
        decimal_places=15,
    ),
]
NominalType = Annotated[
    Decimal,
    Field(
        gt=-999999999999999.999999999999999,
        lt=999999999999999.999999999999999,
        max_digits=30,
        decimal_places=15,
    ),
]


class SoftDeleteMixin:
    deleted_at: datetime | None


class SoftDeleteFilterMixin:
    deleted_at__isnull: bool | None = None


class TaskMixin:
    status: enums.TaskStatus
    message: str | None
    progress: Annotated[float | None, AfterValidator(validate_zero_to_one)]
    started_at: datetime | None
    ended_at: datetime | None


class TaskFilterMixin:
    status: enums.TaskStatus | None = None
