import subprocess
import sys
from pathlib import Path

from loguru import logger


EXCLUDE_DIRS = {
    ".venv",
    "venv",
    "env",
    "__pypackages__",
    "build",
    "dist",
    "hooks",
    "migrations",
}


def is_excluded(path: Path):
    """Check if the path is inside excluded directories or *.egg-info"""
    for part in path.parts:
        if part in EXCLUDE_DIRS or part.endswith(".egg-info"):
            return True
    return False


def find_project_root() -> Path:
    """Walk upward to find the directory containing pyproject.toml"""
    current = Path.cwd()
    for parent in [current] + list(current.parents):
        if (parent / "pyproject.toml").is_file():
            return parent
    logger.info("❌ Could not find pyproject.toml in any parent directory.")
    sys.exit(1)


def find_python_files(directory: Path):
    """Find all .py files recursively, excluding virtual envs and egg-info"""
    return [p for p in directory.rglob("*.py") if p.is_file() and not is_excluded(p)]


def run_black(files):
    """Run black formatter on a list of files"""
    try:
        subprocess.run(["black", *map(str, files)], check=True)
    except FileNotFoundError:
        logger.info("❌ 'black' not found. Install it with: pip install black")
        sys.exit(1)
    except subprocess.CalledProcessError as e:
        logger.info(f"❌ Black failed with exit code {e.returncode}")
        sys.exit(e.returncode)


def run_pylint(files):
    """Run pylint on a list of files, using .pylintrc from working directory if present"""
    pylintrc = Path.cwd() / ".pylintrc"
    cmd = ["pylint"]
    if pylintrc.is_file():
        cmd += ["--rcfile", str(pylintrc)]
    cmd += list(map(str, files))
    try:
        result = subprocess.run(
            cmd,
            check=False,
        )
        if result.returncode != 0:
            logger.info(f"❌ Pylint found issues (exit code {result.returncode})")
            sys.exit(result.returncode)
    except FileNotFoundError:
        logger.info("❌ 'pylint' not found. Install it with: pip install pylint")
        sys.exit(1)


def main():
    project_root = find_project_root()
    logger.info(f"📂 Project root detected: {project_root}")
    py_files = find_python_files(project_root)

    if not py_files:
        logger.info("✅ No Python files found to format or lint.")
        return

    logger.info(f"🎯 Formatting {len(py_files)} Python file(s)...")
    run_black(py_files)
    logger.info("✅ Formatting complete.")

    logger.info("🔎 Running pylint for code analysis...")
    run_pylint(py_files)
    logger.info("✅ Pylint check complete.")


if __name__ == "__main__":
    main()
