from email import policy
from email.message import EmailMessage
from email.parser import BytesParser
from smtplib import SMTP, SMTPNotSupportedError

import requests
from jinja2 import Environment, FileSystemLoader
from loguru import logger

from .settings import MailingSettings, core_settings


class EmailClient:
    def __init__(
        self,
        settings: MailingSettings,
        template_dir: str = None,
        test_mode: bool = False,
        debug: bool = False,
    ):
        self._settings = settings
        self._template_env = Environment(
            loader=FileSystemLoader(template_dir or "templates/email")
        )
        self._test_mode = test_mode
        self._debug = debug

    @property
    def settings(self) -> MailingSettings:
        return self._settings

    @property
    def template_env(self) -> Environment:
        return self._template_env

    @property
    def test_mode(self) -> bool:
        return self._test_mode

    @property
    def debug(self) -> bool:
        return self._debug

    @classmethod
    def load_eml_file(cls, filename: str) -> EmailMessage:
        with open(filename, "rb") as f:
            return BytesParser(policy=policy.default).parse(f)

    async def send_message(self, message: EmailMessage):
        if self.test_mode:
            filename = f"tests/tempfiles/{message["To"].replace('@', '_').replace('.', '_')}.eml"
            with open(filename, "wb") as f:
                f.write(message.as_bytes(policy=policy.default))
            return
        await getattr(self, f"_handle_{self.settings.smtp_mode}_send")(message)

    async def _handle_gmail_send(self, message: EmailMessage):
        with SMTP(self.settings.smtp_host, self.settings.smtp_port) as smtp:
            if self.debug:
                smtp.set_debuglevel(1)
            smtp.starttls()
            smtp.login(self.settings.smtp_username, self.settings.smtp_password)
            smtp.send_message(message)

    async def _handle_o365_send(self, message: EmailMessage):
        response = requests.post(
            f"https://login.microsoftonline.com/{self.settings.smtp_tenant_id}/oauth2/v2.0/token",
            data={
                "grant_type": "client_credentials",
                "client_id": self.settings.smtp_client_id,
                "client_secret": self.settings.smtp_client_secret,
                "scope": "https://graph.microsoft.com/.default",
            },
            timeout=10,
        )
        response.raise_for_status()
        access_token = response.json()["access_token"]
        with SMTP(self.settings.smtp_host, self.settings.smtp_port) as smtp:
            if self.debug:
                smtp.set_debuglevel(1)
            smtp.ehlo(core_settings.host)
            smtp.starttls()
            smtp.ehlo(core_settings.host)
            smtp.login(self.settings.smtp_username, access_token)
            smtp.send_message(message)

    async def _handle_smtp_send(self, message: EmailMessage):
        with SMTP(self.settings.smtp_host, self.settings.smtp_port) as smtp:
            if self.debug:
                smtp.set_debuglevel(1)
            try:
                smtp.starttls()
            except SMTPNotSupportedError as e:
                if self.debug:
                    logger.info(f"Failed to start TLS: {e}")
                    logger.info("Continuing without TLS")
            if (self.settings.smtp_username is not None) and (
                self.settings.smtp_password is not None
            ):
                smtp.login(self.settings.smtp_username, self.settings.smtp_password)
            smtp.send_message(message)

    # pylint: disable=too-many-arguments,too-many-positional-arguments
    async def get_templated_message(
        self,
        template_name: str,
        subject: str,
        to: str,
        body: dict,
        send_from: str = None,
    ) -> EmailMessage:
        template = self.template_env.get_template(template_name)
        html_content = template.render(**body)

        message = EmailMessage()
        message["Subject"] = subject
        message["From"] = (
            send_from or f"{self.settings.smtp_from_name} <{self.settings.smtp_from}>"
        )
        message["To"] = to
        message.set_content(html_content, subtype="html")
        return message

    # pylint: enable=too-many-arguments,too-many-positional-arguments

    async def get_html_preview(
        self,
        template_name: str,
        body: dict,
    ) -> str:
        template = self.template_env.get_template(template_name)
        return template.render(**body)
