import logging
import sys

from gunicorn.glogging import Logger
from loguru import logger

from .settings import core_settings

LOG_LEVEL = "DEBUG" if core_settings.debug else "INFO"


# pylint: disable=protected-access
class InterceptHandler(logging.Handler):
    def emit(self, record):
        try:
            level = logger.level(record.levelname).name
        except ValueError:
            level = record.levelno

        frame, depth = sys._getframe(6), 6
        while frame and frame.f_code.co_filename == logging.__file__:
            frame = frame.f_back
            depth += 1

        logger.opt(depth=depth, exception=record.exc_info).log(
            level, record.getMessage()
        )


# pylint: enable=protected-access


class StubbedGunicornLogger(Logger):
    def setup(self, cfg):
        handler = logging.NullHandler()
        self.error_logger = logging.getLogger("gunicorn.error")
        self.error_logger.addHandler(handler)
        self.error_logger.setLevel(LOG_LEVEL)
        self.access_logger = logging.getLogger("gunicorn.access")
        self.access_logger.addHandler(handler)
        self.access_logger.setLevel(LOG_LEVEL)


def setup_logging():
    logger.remove()

    intercept_handler = InterceptHandler()
    logging.root.setLevel(LOG_LEVEL)

    seen = set()
    for name in [
        # pylint: disable=no-member
        *logging.root.manager.loggerDict.keys(),
        # pylint: enable=no-member
        "gunicorn",
        "gunicorn.access",
        "gunicorn.error",
        "uvicorn",
        "uvicorn.access",
        "uvicorn.error",
        "celery_tasks.log",
    ]:
        if name not in seen:
            seen.add(name.split(".")[0])
            logging.getLogger(name).handlers = [intercept_handler]

    handlers = [
        {
            "sink": sys.stdout,
            "serialize": False,
            "enqueue": True,
            "backtrace": True,
            "diagnose": core_settings.debug,
        }
    ]

    if core_settings.dump_logs:
        handlers.append(
            {
                "sink": ".logs/be.log",
                "serialize": False,
                "rotation": "00:00",
                "retention": "90 days",
                "enqueue": True,
                "backtrace": True,
                "diagnose": core_settings.debug,
            }
        )

    logger.configure(handlers=handlers)
