import re

from fastapi import Request
from fastapi.responses import JSONResponse
from loguru import logger
from sqlalchemy.exc import IntegrityError

from ..settings import core_settings


async def get_unique_violation_key(
    exc: IntegrityError, pattern: str = r"Duplicate entry '([^']+)'"
):
    match = re.search(pattern, str(exc.orig))
    if not match:
        return None
    key = match.group(1)
    return key


async def get_fk_violation_key(
    exc: IntegrityError, pattern: str = r"FOREIGN KEY \(`([^`]+)`\)"
):
    match = re.search(pattern, str(exc.orig))
    if not match:
        return None
    key = match.group(1)
    return key


async def generic_exception_handler(request: Request, exc: Exception):
    logger.error("An unknown error occurred.")
    if core_settings.debug:
        logger.debug(request)
        logger.exception(exc)
    return JSONResponse({"detail": "An unknown error occurred"}, status_code=500)


async def fk_violation_exception_handler(_, exc):
    key = await get_fk_violation_key(exc)
    if "Cannot delete or update a parent row:" in str(exc.orig):
        return JSONResponse(
            {"detail": "This object is still referenced by another object(s)."},
            status_code=400,
        )
    return JSONResponse({"detail": f'"{key}" does not exist.'}, status_code=400)


async def unique_violation_exception_handler(_, exc):
    key = await get_unique_violation_key(exc)
    return JSONResponse({"detail": f'"{key}" already exists.'}, status_code=400)


async def integrity_exception_handler(request: Request, exc: IntegrityError):
    logger.error("An unknown data integrity error occurred.")
    if core_settings.debug:
        logger.exception(exc)
    exc_string = str(exc.orig)
    if "foreign key constraint fails" in exc_string:
        return await fk_violation_exception_handler(request, exc)
    if "Duplicate entry" in exc_string:
        return await unique_violation_exception_handler(request, exc)
    return JSONResponse(
        {"detail": "An unknown data integrity error occurred."}, status_code=400
    )
