import sqlalchemy
from loguru import logger
from sqlalchemy import inspect

from .settings import core_settings


def get_db_url():
    # Build the database URL from core_settings attributes
    settings = core_settings
    if not all(
        [
            settings.db_host,
            settings.db_port,
            settings.db_database,
            settings.db_username,
            settings.db_password,
        ]
    ):
        return None
    return (
        f"mysql+pymysql://{settings.db_username}:{settings.db_password}"
        f"@{settings.db_host}:{settings.db_port}/{settings.db_database}"
    )


def get_dbml_type(sql_type):
    mapping = {
        "INTEGER": "int",
        "SMALLINT": "int",
        "BIGINT": "int",
        "VARCHAR": "varchar",
        "TEXT": "text",
        "DATE": "date",
        "DATETIME": "datetime",
        "TIMESTAMP": "timestamp",
        "BOOLEAN": "bool",
        "FLOAT": "float",
        "NUMERIC": "decimal",
        "DECIMAL": "decimal",
    }
    return mapping.get(sql_type.upper(), sql_type.lower())


def main():
    db_url = get_db_url()
    if not db_url:
        logger.info("Database settings are incomplete in core_settings")
        return

    engine = sqlalchemy.create_engine(db_url)
    inspector = inspect(engine)
    dbml = []
    refs = []

    for table_name in inspector.get_table_names():
        dbml.append(f"Table {table_name} {{")
        for column in inspector.get_columns(table_name):
            col_type = str(column["type"]).split("(")[0]
            dbml_type = get_dbml_type(col_type)
            pk = " [pk]" if column.get("primary_key", False) else ""
            nullable = "" if column.get("nullable", True) else " [not null]"
            dbml.append(f"  {column['name']} {dbml_type}{pk}{nullable}")
        dbml.append("}\n")

        # Add foreign key relationships
        for fk in inspector.get_foreign_keys(table_name):
            if not fk.get("referred_table") or not fk.get("constrained_columns"):
                continue
            # Format: Ref: table.column > referred_table.referred_column
            for col, ref_col in zip(fk["constrained_columns"], fk["referred_columns"]):
                refs.append(
                    f"Ref: {table_name}.{col} > {fk['referred_table']}.{ref_col}"
                )

    logger.info("\n".join(dbml))
    if refs:
        logger.info("\n" + "\n".join(refs))


if __name__ == "__main__":
    main()
