import pickle
from typing import Any, Awaitable

from redis import Redis
from redis.asyncio import Redis as AsyncRedis
from .settings import RedisSettings


class BaseCache:
    def __init__(self, app: str, settings: RedisSettings):
        self._settings = settings
        self._app = app

    @property
    def settings(self) -> RedisSettings:
        return self._settings

    @property
    def app(self) -> str:
        return self._app

    def construct_key(self, *args) -> str:
        return ":".join([self.app] + [str(arg) for arg in args])


class AsyncCache(BaseCache):
    def __init__(self, app: str, settings: RedisSettings):
        super().__init__(app, settings)
        self._session = AsyncRedis(
            host=self._settings.host,
            port=self._settings.port,
            db=self._settings.database,
        )

    @property
    def session(self) -> AsyncRedis:
        return self._session

    # pylint: disable=too-many-arguments,too-many-positional-arguments
    async def aset(
        self,
        value: Any,
        module: str,
        submodule: str,
        scope: str,
        ident: str = "null",
        ttl: int = None,
        keepttl: bool = False,
    ) -> Awaitable[Any]:
        key = self.construct_key(module, submodule, scope, ident)
        payload = pickle.dumps(value)
        return await self._session.set(key, payload, ex=ttl, keepttl=keepttl)

    # pylint: enable=too-many-arguments,too-many-positional-arguments

    async def aget(
        self, module: str, submodule: str, scope: str, ident: str = "null"
    ) -> Awaitable[Any]:
        key = self.construct_key(module, submodule, scope, ident)
        response = await self._session.get(key)
        return pickle.loads(response)

    async def adelete(
        self, module: str, submodule: str, scope: str, ident: str = "null"
    ) -> Awaitable[Any]:
        key = self.construct_key(module, submodule, scope, ident)
        return await self._session.delete(key)

    async def aexists(
        self, module: str, submodule: str, scope: str, ident: str = "null"
    ) -> Awaitable[bool]:
        key = self.construct_key(module, submodule, scope, ident)
        res = await self._session.exists(key)
        return res == 1

    async def aclose(self):
        await self._session.close()

    def flush(self):
        self._session.flushdb()


class SyncCache(BaseCache):
    def __init__(self, app: str, settings: RedisSettings):
        super().__init__(app, settings)
        self._session = Redis(
            host=self._settings.host,
            port=self._settings.port,
            db=self._settings.database,
        )

    @property
    def session(self) -> Redis:
        return self._session

    # pylint: disable=too-many-arguments,too-many-positional-arguments
    def set(
        self,
        value: Any,
        module: str,
        submodule: str,
        scope: str,
        ident: str = "null",
        ttl: int = None,
        keepttl: bool = False,
    ) -> Any:
        key = self.construct_key(module, submodule, scope, ident)
        payload = pickle.dumps(value)
        return self._session.set(key, payload, ex=ttl, keepttl=keepttl)

    # pylint: enable=too-many-arguments,too-many-positional-arguments

    def get(self, module: str, submodule: str, scope: str, ident: str = "null") -> Any:
        key = self.construct_key(module, submodule, scope, ident)
        response = self._session.get(key)
        return pickle.loads(response)

    def delete(
        self, module: str, submodule: str, scope: str, ident: str = "null"
    ) -> Any:
        key = self.construct_key(module, submodule, scope, ident)
        return self._session.delete(key)

    def exists(
        self, module: str, submodule: str, scope: str, ident: str = "null"
    ) -> bool:
        key = self.construct_key(module, submodule, scope, ident)
        res = self._session.exists(key)
        return res == 1

    def close(self):
        self._session.close()

    def flush(self):
        self._session.flushdb()


class CacheConfig:
    def __init__(self, app: str, settings: RedisSettings):
        self._settings = settings
        self._app = app

    def get_sync_cache(self) -> SyncCache:
        return SyncCache(self._app, self._settings)

    def get_async_cache(self) -> AsyncCache:
        return AsyncCache(self._app, self._settings)
