from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware
from fastapi.middleware.gzip import GZipMiddleware
from gunicorn.app.base import BaseApplication
from sqlalchemy.exc import IntegrityError

from ..exceptions import handlers
from ..logging import setup_logging
from ..settings import CoreSettings
from .options import get_options


# pylint: disable=abstract-method
class Application(BaseApplication):
    def __init__(self, app: FastAPI | str, options: dict):
        self.options = options
        self.application = app
        super().__init__()

    def load_config(self):
        for key, value in self.options.items():
            if key in self.cfg.settings and value is not None:
                self.cfg.set(key.lower(), value)

    def load(self):
        return self.application


# pylint: enable=abstract-method


def get_app(settings: CoreSettings):
    app = FastAPI(
        debug=settings.debug,
        title=settings.service_name,
        version=settings.version,
        redoc_url=None,
        docs_url=f"/docs/{settings.service_shortname}" if settings.debug else None,
        openapi_url=("/openapi.json" if settings.debug else None),
        root_path=f"/api/{settings.service_shortname}",
    )

    # Middleware is reverse-ordered
    app.add_middleware(GZipMiddleware)
    app.add_middleware(
        CORSMiddleware,
        allow_origins=settings.allowed_origins or ["*"],
        allow_credentials=True,
        allow_methods=["*"],
        allow_headers=["*"],
    )

    app.add_exception_handler(Exception, handlers.generic_exception_handler)
    app.add_exception_handler(IntegrityError, handlers.integrity_exception_handler)

    return app


def run_app(app: FastAPI | str, settings: CoreSettings, binary_mode: bool = False):
    setup_logging()

    options = get_options(settings, binary_mode)

    Application(app, options).run()
