import io

import pandas as pd
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User

from be_accounting.coa.enums import AccountTypeEnum
from ..coa.repositories import AccountRepository
from . import enums, schemas, repositories


async def create_entry(
    db: AsyncSession,
    entry: schemas.EntryCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.EntryRepository(db)
    item_repo = repositories.EntryItemRepository(db)
    entry_data = entry.model_dump()
    entry_items = entry_data.pop("items")
    obj = await repo.acreate(
        **entry_data,
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
        _commit=False,
    )
    for entry_item in entry_items:
        entry_item.update(
            {
                "entry_id": obj.pk,
            }
        )
        await item_repo.acreate(**entry_item, _commit=False)
    await db.commit()
    await db.refresh(obj)
    return obj


async def retrieve_entry(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.EntryRepository(db, organization_id=group.organization_id)
    obj = await repo.aget_or_404(pk)
    return obj


async def update_entry(
    db: AsyncSession,
    pk: int,
    entry: schemas.EntryUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.EntryRepository(db, organization_id=group.organization_id)
    obj = await repo.aupdate(pk, **entry.model_dump(exclude_unset=True))
    return obj

async def approve_entry(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.EntryRepository(db, organization_id=group.organization_id)
    obj = await repo.aupdate(pk, status=enums.EntryStatusEnum.APPROVED)
    return obj


async def list_entries(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.EntryFilter,
    ordering: list[enums.EntryOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.EntryRepository(db, organization_id=group.organization_id)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def delete_entry(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.EntryRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)


async def record_entry(
    db: AsyncSession,
    entry: schemas.EntryCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.EntryRepository(db)
    item_repo = repositories.EntryItemRepository(db)
    entry_data = entry.model_dump()
    entry_items = entry_data.pop("items")
    obj = await repo.acreate(
        **entry_data,
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
        status=enums.EntryStatusEnum.POSTED,
        _commit=False,
    )
    for entry_item in entry_items:
        entry_item.update(
            {
                "entry_id": obj.pk,
            }
        )
        entry_item_obj = await item_repo.acreate(**entry_item, _commit=False)
        account_repo = AccountRepository(db, organization_id=group.organization_id)
        account = await account_repo.aget_or_404(entry_item_obj.account_id)
        if account.account_type in { AccountTypeEnum.ASSET, AccountTypeEnum.EXPENSE, }:
            if entry_item_obj.side == enums.EntrySide.DEBIT:
                account.balance += entry_item_obj.amount
            elif entry_item_obj.side == enums.EntrySide.CREDIT:
                account.balance -= entry_item_obj.amount
        elif account.account_type in { AccountTypeEnum.LIABILITY, AccountTypeEnum.EQUITY, AccountTypeEnum.REVENUE, }:
            if entry_item_obj.side == enums.EntrySide.DEBIT:
                account.balance -= entry_item_obj.amount
            elif entry_item_obj.side == enums.EntrySide.CREDIT:
                account.balance += entry_item_obj.amount
        await account_repo.aupdate(account.pk, balance=account.balance, _commit=False)
    await db.commit()
    await db.refresh(obj)
    return obj
