from typing import Literal
from typing_extensions import Annotated
from pydantic import PositiveInt, StringConstraints, field_validator, model_validator

from be_kit.paginations import BasePaginatedResponse
from be_kit.schemas import BaseSchema, BaseORMSchema, NominalType
from be_core_services.country.schemas import CurrencyMin
from be_uam.kit.schemas import MetadataMixin
from be_uam.user.schemas import UserMin
from ..coa.schemas import AccountMin
from . import enums


class EntryItemBase:
    side: enums.EntrySide
    amount: NominalType


class EntryItemCreate(EntryItemBase, BaseSchema):
    account_id: PositiveInt


class EntryItem(EntryItemBase, BaseORMSchema):
    account: AccountMin


class EntryCreate(BaseSchema):
    description: Annotated[str | None, StringConstraints(max_length=500)]
    currency_id: PositiveInt
    entry_type: enums.EntryTypeEnum | None = None
    items: list[EntryItemCreate]

    @field_validator("items")
    @classmethod
    def validate_items_not_empty(cls, v):
        if not v or len(v) < 2:
            raise ValueError("At least two items (debit and credit) are required.")
        return v

    @model_validator(mode="after")
    def validate_debit_credit_balance(self):
        debit_total = sum(
            item.amount for item in self.items if item.side == enums.EntrySide.DEBIT
        )
        credit_total = sum(
            item.amount for item in self.items if item.side == enums.EntrySide.CREDIT
        )
        if debit_total == 0 or credit_total == 0:
            raise ValueError("There must be at least one debit and one credit item.")
        if debit_total != credit_total:
            raise ValueError("Total debit and credit amounts must be equal.")
        return self


class EntryUpdate(BaseSchema):
    status: Literal[enums.EntryStatusEnum.POSTED, enums.EntryStatusEnum.CANCELLED]


class Entry(MetadataMixin, BaseORMSchema):
    identifier: Annotated[str, StringConstraints(max_length=10)]
    description: Annotated[str | None, StringConstraints(max_length=500)]
    currency: CurrencyMin
    entry_type: enums.EntryTypeEnum | None = None
    items: list[EntryItem]


class PaginatedEntry(BasePaginatedResponse):
    items: list[Entry]


class EntryFilter(BaseSchema):
    identifier: str | None = None
    identifier__icontains: str | None = None
    currency_id: PositiveInt | None = None
    entry_type: enums.EntryTypeEnum | None = None


class EntryMin(BaseORMSchema):
    pass


class EntryOpt(EntryMin):
    pass


class PaginatedEntryOpt(BasePaginatedResponse):
    items: list[EntryOpt]


UserMin.model_rebuild()
