from datetime import date
from decimal import Decimal

from sqlalchemy import (
    BigInteger,
    Enum,
    ForeignKey,
    String,
    UniqueConstraint,
    Date,
)
from sqlalchemy.orm import Mapped, mapped_column, relationship

from be_kit.databases import BaseModel, SoftDeleteMixin, NominalField
from be_uam.kit.models import MetadataMixin, OrganizationMixin
from be_core_services.country.models import Currency
from ..coa.models import Account
from . import enums


class Entry(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "accounting_journal_entry"

    status: Mapped[enums.EntryStatusEnum] = mapped_column(
        Enum(enums.EntryStatusEnum), nullable=False, default=enums.EntryStatusEnum.DRAFT
    )
    currency_id: Mapped[int] = mapped_column(
        ForeignKey("core_services_currency.pk", ondelete="RESTRICT"),
    )
    currency: Mapped[Currency] = relationship(
        "Currency",
        passive_deletes=True,
        lazy="selectin",
    )
    items: Mapped[list["EntryItem"]] = relationship(
        "EntryItem",
        back_populates="entry",
        cascade="all, delete-orphan",
        lazy="selectin",
    )
    description: Mapped[str | None] = mapped_column(String(500), nullable=True)
    identifier: Mapped[str] = mapped_column(String(10), unique=True, index=True)
    entry_type: Mapped[enums.EntryTypeEnum] = mapped_column(
        Enum(enums.EntryTypeEnum), nullable=True,
    )


class EntryItem(BaseModel):
    __tablename__ = "accounting_journal_entry_item"
    __table_args__ = (UniqueConstraint("entry_id", "account_id"),)

    entry_id: Mapped[int] = mapped_column(
        ForeignKey("accounting_journal_entry.pk", ondelete="CASCADE"),
    )
    entry: Mapped[Entry] = relationship(
        "Entry",
        back_populates="items",
        passive_deletes=True,
        lazy="selectin",
    )
    account_id: Mapped[int] = mapped_column(
        ForeignKey("accounting_coa_account.pk", ondelete="RESTRICT"),
    )
    account: Mapped[Account] = relationship(
        "Account",
        passive_deletes=True,
        lazy="selectin",
    )
    side: Mapped[enums.EntrySide] = mapped_column(
        Enum(enums.EntrySide),
        nullable=False,
    )
    amount: Mapped[Decimal] = mapped_column(NominalField, nullable=False)


class Counter(OrganizationMixin, BaseModel):
    __tablename__ = "accounting_journal_counter"
    __table_args__ = (UniqueConstraint("organization_id", "counter_date"),)

    counter_date: Mapped[date] = mapped_column(Date, index=True)
    counter: Mapped[int] = mapped_column(BigInteger, default=0)
