from typing import Annotated

from fastapi import APIRouter, Depends, Query, status
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user, authorize
from ..databases import get_async_session
from . import enums, schemas, utils
from .permissions import permission_mapper


router = APIRouter(tags=["entries"], prefix="/entry")


@router.post(
    "/",
    dependencies=[Depends(authorize(permission_mapper["create_entry"]))],
    response_model=schemas.Entry,
    status_code=status.HTTP_201_CREATED,
)
async def create_entry(
    entry: schemas.EntryCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_entry(session, entry, request_user)


@router.get(
    "/{entry_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["retrieve_entry"]))],
    response_model=schemas.Entry,
)
async def retrieve_entry(
    entry_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_entry(session, entry_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(permission_mapper["list_entry"]))],
    response_model=schemas.PaginatedEntry,
)
async def list_entry(
    pagination: PaginationQuery = Depends(),
    filters: schemas.EntryFilter = Depends(),
    ordering: Annotated[list[enums.EntryOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_entries(
        session, pagination, filters, ordering, request_user
    )


@router.put(
    "/{entry_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["update_entry"]))],
    response_model=schemas.Entry,
)
async def update_entry(
    entry_id: int,
    entry: schemas.EntryUpdate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_entry(session, entry_id, entry, request_user)


@router.get(
    "/options/",
    dependencies=[Depends(authorize(permission_mapper["list_entry_option"]))],
    response_model=schemas.PaginatedEntryOpt,
)
async def list_entry_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.EntryFilter = Depends(),
    ordering: Annotated[list[enums.EntryOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_entries(
        session, pagination, filters, ordering, request_user
    )


@router.post(
    "/{entry_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["approve_entry"]))],
    response_model=schemas.Entry,
)
async def approve_entry(
    entry_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.approve_entry(session, entry_id, request_user)


@router.delete(
    "/{entry_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["delete_entry"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_entry(
    entry_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_entry(session, entry_id, request_user)
