from decimal import Decimal
import io
import os

import pandas as pd
from sqlalchemy import select
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from . import enums, schemas, repositories
from pathlib import Path

# Account Category


async def create_account_category(
    db: AsyncSession,
    account_category: schemas.AccountCategoryCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountCategoryRepository(db)
    obj = await repo.acreate(
        **account_category.model_dump(),
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
    )
    return obj


async def retrieve_account_category(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountCategoryRepository(
        db, organization_id=group.organization_id
    )
    obj = await repo.aget_or_404(pk)
    return obj


async def update_account_category(
    db: AsyncSession,
    pk: int,
    account_category: schemas.AccountCategoryUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountCategoryRepository(
        db, organization_id=group.organization_id
    )
    obj = await repo.aupdate(pk, **account_category.model_dump())
    return obj


async def list_account_categories(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.AccountCategoryFilter,
    ordering: list[enums.AccountCategoryOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountCategoryRepository(
        db, organization_id=group.organization_id
    )
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def delete_account_category(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountCategoryRepository(
        db, organization_id=group.organization_id
    )
    await repo.adelete(pk)

# Account

async def create_account(
    db: AsyncSession,
    account: schemas.AccountCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountRepository(db)
    obj = await repo.acreate(
        **account.model_dump(),
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
    )
    return obj


async def retrieve_account(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountRepository(db, organization_id=group.organization_id)
    obj = await repo.aget_or_404(pk)
    return obj


async def list_accounts(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.AccountFilter,
    ordering: list[enums.AccountOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountRepository(db, organization_id=group.organization_id)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def update_account(
    db: AsyncSession,
    pk: int,
    account: schemas.AccountUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountRepository(db, organization_id=group.organization_id)
    obj = await repo.aupdate(
        pk, **account.model_dump(), last_modified_by_id=request_user.pk
    )
    return obj


async def delete_account(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)


async def handle_account_upload(
    db: AsyncSession,
    account_upload: schemas.AccountUpload,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountRepository(db, organization_id=group.organization_id)

    content = await account_upload.file.read()
    df = pd.read_excel(io.BytesIO(content))

    required_columns = {"account_number", "name", "currency_id"}
    if not required_columns.issubset(df.columns):
        missing = required_columns - set(df.columns)
        raise ValueError(f"Missing required columns: {', '.join(missing)}")

    created_accounts = []
    for _, row in df.iterrows():
        account_data = {
            "account_number": str(row["account_number"]),
            "name": str(row["name"]),
            "currency_id": int(row["currency_id"]),
        }
        obj = await repo.acreate(
            **account_data,
            organization_id=group.organization_id,
            created_by_id=request_user.pk,
        )
        created_accounts.append(schemas.AccountMin.model_validate(obj).model_dump())

    return created_accounts


async def handle_account_download(
    db: AsyncSession,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountRepository(db, organization_id=group.organization_id)
    accounts = await repo.arecords()

    account_rows = []
    for account in accounts:
        account_rows.append(
            {
                "account_number": account.account_number,
                "name": account.name,
                "currency_id": account.currency_id,
            }
        )

    df = pd.DataFrame(account_rows)
    output = io.BytesIO()
    with pd.ExcelWriter(output, engine="openpyxl") as writer:
        df.to_excel(writer, sheet_name="account", index=False)
    output.seek(0)
    return output


# Account Mapping


async def create_account_mapping(
    db: AsyncSession,
    account_mapping: schemas.AccountMappingCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountMappingRepository(db)
    obj = await repo.acreate(
        **account_mapping.model_dump(),
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
    )
    return obj


async def retrieve_account_mapping(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountMappingRepository(
        db, organization_id=group.organization_id
    )
    obj = await repo.aget_or_404(pk)
    return obj


async def list_account_mappings(
    db: AsyncSession,
    pagination: PaginationQuery,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountMappingRepository(
        db, organization_id=group.organization_id
    )
    objs = await repo.arecords(pagination=pagination)
    return objs


async def update_account_mapping(
    db: AsyncSession,
    pk: int,
    account_mapping: schemas.AccountMappingUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountMappingRepository(
        db, organization_id=group.organization_id
    )
    obj = await repo.aupdate(
        pk,
        **account_mapping.model_dump(),
        last_modified_by_id=request_user.pk,
    )
    return obj


async def delete_account_mapping(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AccountMappingRepository(
        db, organization_id=group.organization_id
    )
    await repo.adelete(pk)

async def list_account_mappings_by_user(
    db: AsyncSession,
    user: User,
):
    group = await user.awaitable_attrs.group
    repo = repositories.AccountMappingRepository(db, organization_id=group.organization_id)
    filters = schemas.AccountMappingFilter(
        organization_id=group.organization_id,
        created_by_id=user.pk
    )
    objs = await repo.arecords(filters, pagination=None, ordering=None)
    objs = objs.all() if hasattr(objs, "all") else objs  # Ensure it's a list
    if not objs:
        return None
    obj = objs[0]
    return schemas.AccountMapping.model_validate(obj)


async def seed_initial_coa(db: AsyncSession, request_user: User):
    group = await request_user.awaitable_attrs.group
    org_id = group.organization_id

    BASE_DIR = Path(__file__).resolve().parent.parent
    EXCEL_PATH = BASE_DIR / "resources" / "init_coa.xlsx"
    df = pd.read_excel(EXCEL_PATH)

    category_repo = repositories.AccountCategoryRepository(db, organization_id=org_id)
    account_repo = repositories.AccountRepository(db, organization_id=org_id)

    created_categories = {}
    created_accounts = []

    for _, row in df.iterrows():
        account_name = str(row["AccountName"]).strip()
        account_number = str(row["AccountNumber"]).strip()
        category_name = str(row["CategoryName"]).strip()
        account_type_str = str(row["Account Type"]).strip().upper()
        desc = row.get("Description", None)
        initial_balance = Decimal(row.get("InitialBalance", 0))

        # Validate enum
        if account_type_str not in enums.AccountTypeEnum.__members__:
            raise ValueError(f"Invalid Account Type '{account_type_str}' in: {account_name}")

        account_type = enums.AccountTypeEnum[account_type_str]

        if category_name not in created_categories:
            existing_category_query = await category_repo.find_by_name_and_org(
                name=category_name,
                organization_id=org_id
            )
            existing_category = existing_category_query

            if existing_category:
                created_categories[category_name] = existing_category.pk
            else:
                new_category = await category_repo.acreate(
                    name=category_name,
                    description=None,
                    organization_id=org_id,
                    created_by_id=request_user.pk,
                )
                created_categories[category_name] = new_category.pk

        category_id = created_categories[category_name]

        # Create account using acreate() — your standard
        existing_account = await account_repo.find_by_number_and_org(
            account_number=account_number,
            organization_id=org_id
        )

        if existing_account:
            account_obj = existing_account
        else:
            account_obj = await account_repo.acreate(
                account_number=account_number,
                name=account_name,
                account_type=account_type,
                category_id=category_id,
                balance=initial_balance,
                currency_id=1,
                organization_id=org_id,
                created_by_id=request_user.pk,
            )
            created_accounts.append(account_obj)

    return {
        "message": "Chart of Accounts seeded successfully",
        "inserted": len(created_accounts)
    }
