from decimal import Decimal
from typing_extensions import Annotated

from fastapi import File, UploadFile
from pydantic import PositiveInt, StringConstraints

from be_kit.paginations import BasePaginatedResponse
from be_kit.schemas import BaseSchema, BaseORMSchema, NominalType
from be_core_services.country.schemas import CurrencyMin
from be_uam.kit.schemas import MetadataMixin
from be_uam.user.schemas import UserMin
from . import enums

# Account Category

# AccountCategory


class BaseAccountCategory:
    name: str
    description: str | None


class AccountCategoryCreate(BaseAccountCategory, BaseSchema):
    pass


class AccountCategoryUpdate(BaseAccountCategory, BaseSchema):
    pass


class AccountCategory(MetadataMixin, BaseAccountCategory, BaseORMSchema):
    pass


class PaginatedAccountCategory(BasePaginatedResponse):
    items: list[AccountCategory]


class AccountCategoryFilter(BaseSchema):
    name: str | None = None
    description: str | None = None


class AccountCategoryMin(BaseAccountCategory, BaseORMSchema):
    pass


class AccountCategoryOpt(AccountCategoryMin):
    pass


class PaginatedAccountCategoryOpt(BasePaginatedResponse):
    items: list[AccountCategoryOpt]



# Account

class BaseAccount:
    account_type: enums.AccountTypeEnum
    account_number: Annotated[str, StringConstraints(min_length=6, max_length=10)]
    name: Annotated[str, StringConstraints(min_length=1, max_length=100)]


class AccountCreate(BaseAccount, BaseSchema):
    balance: Decimal
    currency_id: PositiveInt
    category_id: PositiveInt | None = None


class AccountUpdate(BaseAccount, BaseSchema):
    currency_id: PositiveInt
    category_id: PositiveInt | None = None


class Account(MetadataMixin, BaseAccount, BaseORMSchema):
    currency: CurrencyMin
    balance: NominalType
    category: AccountCategoryMin | None = None


class PaginatedAccount(BasePaginatedResponse):
    items: list[Account]


class AccountFilter(BaseSchema):
    account_type: enums.AccountTypeEnum | None = None
    name: str | None = None
    name__icontains: str | None = None
    account_number: str | None = None
    account_number__icontains: str | None = None
    currency_id: PositiveInt | None = None
    category_id: PositiveInt | None = None


class AccountMin(BaseAccount, BaseORMSchema):
    pass


class AccountOpt(AccountMin):
    pass


class PaginatedAccountOpt(BasePaginatedResponse):
    items: list[AccountOpt]


class AccountUpload(BaseSchema):
    file: Annotated[UploadFile, File()]


class AccountValidator(BaseSchema):
    account_type: enums.AccountTypeEnum
    account_number: Annotated[str, StringConstraints(min_length=1, max_length=20)]
    name: Annotated[str, StringConstraints(min_length=1, max_length=100)]
    currency_id: PositiveInt


# Account Mapping


class BaseAccountMappingWrite:
    outstanding_receipts_account_id: PositiveInt
    outstanding_payments_account_id: PositiveInt
    sales_advance_account_id: PositiveInt
    cost_of_revenue_account_id: PositiveInt
    account_receivable_account_id: PositiveInt
    account_payable_account_id: PositiveInt
    revenue_account_id: PositiveInt
    vat_out_account_id: PositiveInt
    vat_in_account_id: PositiveInt
    advance_account_id: PositiveInt
    sales_return_account_id: PositiveInt
    default_inventory_account_id: PositiveInt
    default_bank_account_account_id: PositiveInt
    sales_discount_account_id: PositiveInt
    purchase_discount_account_id: PositiveInt
    goods_received_invoice_received_account_id: PositiveInt


class AccountMappingCreate(BaseAccountMappingWrite, BaseSchema):
    pass


class AccountMappingUpdate(BaseAccountMappingWrite, BaseSchema):
    pass


class AccountMapping(MetadataMixin, BaseORMSchema):
    outstanding_receipts_account: AccountMin
    outstanding_payments_account: AccountMin
    sales_advance_account: AccountMin
    cost_of_revenue_account: AccountMin
    account_receivable_account: AccountMin
    account_payable_account: AccountMin
    revenue_account: AccountMin
    vat_out_account: AccountMin
    vat_in_account: AccountMin
    advance_account: AccountMin
    sales_return_account: AccountMin
    default_inventory_account: AccountMin
    default_bank_account_account: AccountMin
    sales_discount_account: AccountMin
    purchase_discount_account: AccountMin
    goods_received_invoice_received_account: AccountMin


class PaginatedAccountMapping(BasePaginatedResponse):
    items: list[AccountMapping]


class AccountMappingFilter(BaseSchema):
    created_by_id: int | None = None
    organization_id: int | None = None


UserMin.model_rebuild()
