from decimal import Decimal

from sqlalchemy import (
    BigInteger,
    Enum,
    ForeignKey,
    String,
    UniqueConstraint,
)
from sqlalchemy.orm import Mapped, mapped_column, relationship

from be_kit.databases import BaseModel, SoftDeleteMixin, NominalField
from be_uam.kit.models import MetadataMixin, OrganizationMixin
from be_core_services.country.models import Currency
from . import enums



class AccountCategory(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "accounting_coa_account_category"
    __table_args__ = (UniqueConstraint("organization_id", "name"),)

    name: Mapped[str] = mapped_column(String(100), index=True)
    description: Mapped[str | None] = mapped_column(String(500), nullable=True)

class Account(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "accounting_coa_account"
    __table_args__ = (UniqueConstraint("organization_id", "account_number"),)

    account_type: Mapped[enums.AccountTypeEnum] = mapped_column(
        Enum(enums.AccountTypeEnum), nullable=False
    )
    account_number: Mapped[str] = mapped_column(String(20))
    name: Mapped[str] = mapped_column(String(100))
    currency_id: Mapped[int] = mapped_column(
        ForeignKey("core_services_currency.pk", ondelete="RESTRICT"),
    )
    currency: Mapped[Currency] = relationship(
        "Currency",
        passive_deletes=True,
        lazy="selectin",
    )
    category_id: Mapped[int] = mapped_column(
        ForeignKey(f"{AccountCategory.__tablename__}.pk", ondelete="RESTRICT"),
        nullable=True,
    )
    category: Mapped[AccountCategory] = relationship(
        "AccountCategory",
        passive_deletes=True,
        lazy="selectin",
    )
    balance: Mapped[Decimal] = mapped_column(NominalField, default=0)


class AccountMapping(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "accounting_coa_account_mapping"
    __table_args__ = (UniqueConstraint("organization_id"),)

    outstanding_receipts_account_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
    )
    outstanding_receipts_account: Mapped[Account] = relationship(
        Account,
        foreign_keys=[outstanding_receipts_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    outstanding_payments_account_id: Mapped[int | None] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
        nullable=True,
    )
    outstanding_payments_account: Mapped[Account | None] = relationship(
        Account,
        foreign_keys=[outstanding_payments_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    sales_advance_account_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
    )
    sales_advance_account: Mapped[Account] = relationship(
        Account,
        foreign_keys=[sales_advance_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    cost_of_revenue_account_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
    )
    cost_of_revenue_account: Mapped[Account] = relationship(
        Account,
        foreign_keys=[cost_of_revenue_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    account_receivable_account_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
    )
    account_receivable_account: Mapped[Account] = relationship(
        Account,
        foreign_keys=[account_receivable_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    account_payable_account_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
    )
    account_payable_account: Mapped[Account] = relationship(
        Account,
        foreign_keys=[account_payable_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    revenue_account_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
    )
    revenue_account: Mapped[Account] = relationship(
        Account,
        foreign_keys=[revenue_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    vat_out_account_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
    )
    vat_out_account: Mapped[Account] = relationship(
        Account,
        foreign_keys=[vat_out_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    vat_in_account_id: Mapped[int | None] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
        nullable=True,
    )
    vat_in_account: Mapped[Account | None] = relationship(
        Account,
        foreign_keys=[vat_in_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    advance_account_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
    )
    advance_account: Mapped[Account] = relationship(
        Account,
        foreign_keys=[advance_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    sales_return_account_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
    )
    sales_return_account: Mapped[Account] = relationship(
        Account,
        foreign_keys=[sales_return_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    default_inventory_account_id: Mapped[int | None] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
        nullable=True,
    )
    default_inventory_account: Mapped[Account | None] = relationship(
        Account,
        foreign_keys=[default_inventory_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    default_bank_account_account_id: Mapped[int | None] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
        nullable=True,
    )
    default_bank_account_account: Mapped[Account | None] = relationship(
        Account,
        foreign_keys=[default_bank_account_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    sales_discount_account_id: Mapped[int | None] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
        nullable=True,
    )
    sales_discount_account: Mapped[Account | None] = relationship(
        Account,
        foreign_keys=[sales_discount_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    purchase_discount_account_id: Mapped[int | None] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
        nullable=True,
    )
    purchase_discount_account: Mapped[Account | None] = relationship(
        Account,
        foreign_keys=[purchase_discount_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
    goods_received_invoice_received_account_id: Mapped[int | None] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="CASCADE"),
        nullable=True,
    )
    goods_received_invoice_received_account: Mapped[Account | None] = relationship(
        Account,
        foreign_keys=[goods_received_invoice_received_account_id],
        passive_deletes=True,
        lazy="selectin",
    )
