import io
from typing import Annotated
from unittest import result

import pandas as pd
from fastapi import APIRouter, Depends, Form, Query, status, UploadFile, File, HTTPException
from fastapi.responses import StreamingResponse
from sqlalchemy.ext.asyncio import AsyncSession
from sqlalchemy import select
from decimal import Decimal

from be_kit.paginations import PaginationQuery
from be_kit.exim.utils import get_version_upload_template
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user, authorize
from ..databases import get_async_session
from . import enums, schemas, utils
from .permissions import account_permission_mapper, account_mapping_permission_mapper, account_category_permission_mapper


# Account

# Account Category

account_category_router = APIRouter(tags=["accounts_categories"], prefix="/account_category")


@account_category_router.post(
    "/",
    dependencies=[
        Depends(
            authorize(account_category_permission_mapper["create_account_category"])
        )
    ],
    response_model=schemas.AccountCategory,
    status_code=status.HTTP_201_CREATED,
)
async def create_account_category(
    account_category: schemas.AccountCategoryCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_account_category(session, account_category, request_user)


@account_category_router.get(
    "/{account_category_id:int}/",
    dependencies=[
        Depends(
            authorize(account_category_permission_mapper["retrieve_account_category"])
        )
    ],
    response_model=schemas.AccountCategory,
)
async def retrieve_account_category(
    account_category_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_account_category(
        session, account_category_id, request_user
    )


@account_category_router.get(
    "/",
    dependencies=[
        Depends(authorize(account_category_permission_mapper["list_account_category"]))
    ],
    response_model=schemas.PaginatedAccountCategory,
)
async def list_account_category(
    pagination: PaginationQuery = Depends(),
    filters: schemas.AccountCategoryFilter = Depends(),
    ordering: Annotated[list[enums.AccountCategoryOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_account_categories(
        session, pagination, filters, ordering, request_user
    )


@account_category_router.get(
    "/options/",
    dependencies=[
        Depends(
            authorize(
                account_category_permission_mapper["list_account_category_option"]
            )
        )
    ],
    response_model=schemas.PaginatedAccountCategoryOpt,
)
async def list_account_category_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.AccountCategoryFilter = Depends(),
    ordering: Annotated[list[enums.AccountCategoryOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_account_categories(
        session, pagination, filters, ordering, request_user
    )


@account_category_router.put(
    "/{account_category_id:int}/",
    dependencies=[
        Depends(
            authorize(account_category_permission_mapper["update_account_category"])
        )
    ],
    response_model=schemas.AccountCategory,
)
async def update_account_category(
    account_category_id: int,
    account_category: schemas.AccountCategoryUpdate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_account_category(
        session, account_category_id, account_category, request_user
    )


@account_category_router.delete(
    "/{account_category_id:int}/",
    dependencies=[
        Depends(
            authorize(account_category_permission_mapper["delete_account_category"])
        )
    ],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_account_category(
    account_category_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_account_category(session, account_category_id, request_user)



# Account

account_router = APIRouter(tags=["accounts"], prefix="/account")


@account_router.post(
    "/",
    dependencies=[Depends(authorize(account_permission_mapper["create_account"]))],
    response_model=schemas.Account,
    status_code=status.HTTP_201_CREATED,
)
async def create_account(
    account: schemas.AccountCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_account(session, account, request_user)


@account_router.get(
    "/{account_id:int}/",
    dependencies=[Depends(authorize(account_permission_mapper["retrieve_account"]))],
    response_model=schemas.Account,
)
async def retrieve_account(
    account_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_account(session, account_id, request_user)


@account_router.get(
    "/",
    dependencies=[Depends(authorize(account_permission_mapper["list_account"]))],
    response_model=schemas.PaginatedAccount,
)
async def list_account(
    pagination: PaginationQuery = Depends(),
    filters: schemas.AccountFilter = Depends(),
    ordering: Annotated[list[enums.AccountOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_accounts(
        session, pagination, filters, ordering, request_user
    )


@account_router.get(
    "/options/",
    dependencies=[Depends(authorize(account_permission_mapper["list_account_option"]))],
    response_model=schemas.PaginatedAccountOpt,
)
async def list_account_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.AccountFilter = Depends(),
    ordering: Annotated[list[enums.AccountOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_accounts(
        session, pagination, filters, ordering, request_user
    )


@account_router.put(
    "/{account_id:int}/",
    dependencies=[Depends(authorize(account_permission_mapper["update_account"]))],
    response_model=schemas.Account,
)
async def update_account(
    account: schemas.AccountUpdate,
    account_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_account(session, account_id, account, request_user)


@account_router.delete(
    "/{account_id:int}/",
    dependencies=[Depends(authorize(account_permission_mapper["delete_account"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_account(
    account_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_account(session, account_id, request_user)


@account_router.get(
    "/upload/template/",
    dependencies=[
        Depends(
            authorize(account_permission_mapper["download_upload_account_template"])
        )
    ],
)
async def download_upload_account_template():
    account_template = await get_version_upload_template(schemas.AccountValidator)

    buffer = io.BytesIO()
    with pd.ExcelWriter(buffer, engine="openpyxl") as writer:
        account_template.to_excel(writer, sheet_name="Account", index=False)
    buffer.seek(0)

    return StreamingResponse(
        buffer,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": 'attachment; filename="Account Template.xlsx"'},
    )


@account_router.post(
    "/upload/",
    dependencies=[Depends(authorize(account_permission_mapper["upload_account"]))],
    status_code=status.HTTP_201_CREATED,
)
async def upload_account(
    account_upload: Annotated[
        schemas.AccountUpload, Form(media_type="multipart/form-data")
    ],
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.handle_account_upload(session, account_upload, request_user)


@account_router.get(
    "/download/",
    dependencies=[Depends(authorize(account_permission_mapper["download_account"]))],
)
async def download_account(
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    buffer = await utils.handle_account_download(session, request_user)
    return StreamingResponse(
        buffer,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": 'attachment; filename="Accounts.xlsx"'},
    )


@account_router.post(
    "/seed_accounts/",
    dependencies=[Depends(authorize(account_permission_mapper["create_account"]))],
    response_model=None,
    status_code=status.HTTP_201_CREATED,
)
async def initialize_chart_of_accounts(
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.seed_initial_coa(session, request_user)


# Account Mapping


account_mapping_router = APIRouter(tags=["account_mappings"], prefix="/account_mapping")


@account_mapping_router.post(
    "/",
    dependencies=[
        Depends(authorize(account_mapping_permission_mapper["create_account_mapping"]))
    ],
    response_model=schemas.AccountMapping,
    status_code=status.HTTP_201_CREATED,
)
async def create_account_mapping(
    account_mapping: schemas.AccountMappingCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_account_mapping(session, account_mapping, request_user)


@account_mapping_router.get(
    "/{account_mapping_id:int}/",
    dependencies=[
        Depends(
            authorize(account_mapping_permission_mapper["retrieve_account_mapping"])
        )
    ],
    response_model=schemas.AccountMapping,
)
async def retrieve_account_mapping(
    account_mapping_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_account_mapping(
        session, account_mapping_id, request_user
    )


@account_mapping_router.get(
    "/",
    dependencies=[
        Depends(authorize(account_mapping_permission_mapper["list_account_mappings"]))
    ],
    response_model=schemas.PaginatedAccountMapping,
)
async def list_account_mappings(
    pagination: PaginationQuery = Depends(),
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_account_mappings(session, pagination, request_user)


@account_mapping_router.get(
    "/user/",
    dependencies=[
        Depends(authorize(account_mapping_permission_mapper["list_account_mappings"]))
    ],
    response_model=schemas.AccountMapping,
)
async def list_account_mappings_by_user(
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_account_mappings_by_user(session, request_user)


@account_mapping_router.put(
    "/{account_mapping_id:int}/",
    dependencies=[
        Depends(authorize(account_mapping_permission_mapper["update_account_mapping"]))
    ],
    response_model=schemas.AccountMapping,
)
async def update_account_mapping(
    account_mapping: schemas.AccountMappingUpdate,
    account_mapping_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_account_mapping(
        session, account_mapping_id, account_mapping, request_user
    )


@account_mapping_router.delete(
    "/{account_mapping_id:int}/",
    dependencies=[
        Depends(authorize(account_mapping_permission_mapper["delete_account_mapping"]))
    ],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_account_mapping(
    account_mapping_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_account_mapping(session, account_mapping_id, request_user)
