from datetime import datetime, timedelta
from decimal import Decimal
import io

from fastapi import HTTPException, UploadFile
from openpyxl import load_workbook
import pandas as pd
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User

from be_accounting.journal.enums import EntrySide
from be_accounting.bank_account.repositories import BankAccountRepository
from be_accounting.bank_reconcile.models import BankStatement
from be_accounting.journal.models import Entry, EntryItem
from be_accounting.coa.utils import list_account_mappings
from be_accounting.journal.repositories import EntryRepository as JournalEntryRepository
from be_accounting.journal import utils as journal_utils, enums as journal_enums, schemas as journal_schemas
from . import enums, schemas, repositories
import re
from dateutil.parser import parse as date_parse
from sqlalchemy import select, and_
from sqlalchemy.orm import selectinload

async def create_bank_statement(
    db: AsyncSession,
    bank_statement: schemas.BankStatementCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankStatementRepository(db)
    obj = await repo.acreate(
        **bank_statement.model_dump(exclude={"lines"}),
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
    )
    # Handle lines if provided
    if bank_statement.lines:
        line_repo = repositories.BankStatementLineRepository(db)
        for line in bank_statement.lines:
            await line_repo.acreate(
                **line.model_dump(),
                bank_statement_id=obj.pk,
                organization_id=group.organization_id,
                created_by_id=request_user.pk,
            )
    return obj


async def retrieve_bank_statement(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankStatementRepository(db, organization_id=group.organization_id)
    obj = await repo.aget_or_404(pk)
    return obj


async def list_bank_statements(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.BankStatementFilter,
    ordering: list[enums.BankStatementOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankStatementRepository(db, organization_id=group.organization_id)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs

async def list_bank_statements_by_bank_id(
    db: AsyncSession,
    bank_account_id: int,
    request_user: User,
    pagination: PaginationQuery,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankStatementRepository(db, organization_id=group.organization_id)
    filters = schemas.BankStatementFilter(bank_account_id=bank_account_id)
    objs = await repo.arecords(filters, pagination, ordering=None)
    return objs


async def list_bank_statement_line_by_statement(
    db: AsyncSession,
    bank_statement_id: int,
    request_user: User,
    pagination: PaginationQuery,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankStatementLineRepository(db, organization_id=group.organization_id)
    filters = schemas.BankStatementLineFilter(bank_statement_id=bank_statement_id)
    objs = await repo.arecords(filters, pagination, ordering=None)
    return objs


async def list_bank_statement_line_by_bank_id(
    db: AsyncSession,
    bank_account_id: int,
    request_user: User,
    pagination: PaginationQuery,
):
    group = await request_user.awaitable_attrs.group

    bank_statement_repo = repositories.BankStatementRepository(db, organization_id=group.organization_id)
    filters = schemas.BankStatementFilter(bank_account_id=bank_account_id)
    bank_statements = await bank_statement_repo.arecords(filters, pagination, ordering=None)

    bank_statement_pks = [row[0] for row in bank_statements.items()]
    repo = repositories.BankStatementLineRepository(db, organization_id=group.organization_id)
    filters = schemas.BankStatementLineFilter(bank_statement_ids=bank_statement_pks)
    objs = await repo.arecords(filters, pagination, ordering=None)
    return objs


async def update_bank_statement(
    db: AsyncSession,
    pk: int,
    bank_statement: schemas.BankStatementUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankStatementRepository(db, organization_id=group.organization_id)
    obj = await repo.aupdate(
        pk, **bank_statement.model_dump(exclude={"lines"}), last_modified_by_id=request_user.pk
    )
    # Handle lines update
    if bank_statement.lines is not None:
        line_repo = repositories.BankStatementLineRepository(db)
        # Delete existing lines
        await line_repo.adelete_by_bank_statement_id(pk)
        # Create new lines
        for line in bank_statement.lines:
            await line_repo.acreate(
                **line.model_dump(),
                bank_statement_id=pk,
                organization_id=group.organization_id,
                created_by_id=request_user.pk,
            )
    return obj


async def delete_bank_statement(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankStatementRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)


async def handle_bank_statement_upload(
    db: AsyncSession,
    bank_statement_upload: schemas.BankStatementUpload,
    bank_account_id: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group


    buffer = await bank_statement_upload.file.read()
    filename = bank_statement_upload.file.filename.lower()

    # Read file using pandas
    try:
        if filename.endswith(".csv"):
            df = pd.read_csv(io.BytesIO(buffer), header=None)
        elif filename.endswith(".xlsx") or filename.endswith(".xls"):
            df = pd.read_excel(io.BytesIO(buffer), header=None)
        else:
            raise HTTPException(400, "Unsupported file format. Only CSV and Excel allowed")
    except Exception:
        raise HTTPException(400, "Failed to parse file. Ensure correct banking format")

    def clean_decimal(value):
        if value is None:
            return Decimal("0")
        value_str = str(value).replace(",", "").strip()
        if not value_str or value_str.upper() in ["NAN", ""]:
            return Decimal("0")
        try:
            return Decimal(value_str)
        except Exception:
            return Decimal("0")

    # Parse the top header section
    header = {}

    # Fetch Bank Code
    bank_account_repo = BankAccountRepository(db, organization_id=group.organization_id)
    bank_account = await bank_account_repo.aget_or_404(bank_account_id)
    bank_code = bank_account.bank.code

    # Initialize repos once (best performance)
    statement_repo = repositories.BankStatementRepository(db, organization_id=group.organization_id)
    line_repo = repositories.BankStatementLineRepository(db, organization_id=group.organization_id)

    print("DEBUG FIRST 10 ROWS:")
    print(df.head(20).to_string())

    if bank_code == "OCBC":
        period_raw = str(df.iloc[0, 1])
        try:
            start_date_str, end_date_str = period_raw.split("-")
            period_start = datetime.strptime(start_date_str.strip(), "%d/%m/%Y").date()
            period_end = datetime.strptime(end_date_str.strip(), "%d/%m/%Y").date()
        except Exception:
            raise HTTPException(status_code=400, detail="Invalid PERIOD format")

        statement_date = datetime.strptime(df.iloc[0, 7].strip(), "%d/%m/%Y").date()
        opening_balance = clean_decimal(df.iloc[1, 7])
        closing_balance = clean_decimal(df.iloc[2, 7])

        total_debit = (df.iloc[3, 1])
        total_credit = (df.iloc[4, 1])

        statement = await statement_repo.acreate(
            bank_account_id=bank_account_id,
            organization_id=group.organization_id,
            created_by_id=request_user.pk,
            statement_date=statement_date,
            opening_balance=opening_balance,
            closing_balance=closing_balance,
            period_start=period_start,
            period_end=period_end,
            total_debit=total_debit,
            total_credit=total_credit,
            status=enums.BankStatementStatus.IMPORTED,
            remark="Imported",
        )

        # Detect transaction table header row
        txn_start_row = None
        for idx, row in df.iterrows():
            if str(row[0]).strip().upper() == "TRANSACTION DATE":
                txn_start_row = idx + 1
                break

        if txn_start_row is None:
            raise HTTPException(status_code=400, detail="Transaction history table not found")

        print("DEBUG TRANSACTION ROWS:")
        print(df.iloc[txn_start_row:txn_start_row+5].to_string())

        # Parse lines
        for idx, row in df.iloc[txn_start_row:].iterrows():
            if pd.isna(row[0]) or not str(row[0]).strip():
                break  # End of data

            # Transaction date conversion
            cell_date = row[0]
            if isinstance(cell_date, datetime):
                transaction_date = cell_date.date()
            else:
                date_str = str(cell_date).strip()
                # Try with time, then fallback to date only
                try:
                    transaction_date = datetime.strptime(date_str, "%d/%m/%Y %H:%M").date()
                except ValueError:
                    transaction_date = datetime.strptime(date_str.split()[0], "%d/%m/%Y").date()

            reference_number = str(row[2]).replace("'", "") if row[2] else None
            description = row[4] or ""

            debit = clean_decimal(row[5])
            credit = clean_decimal(row[6])

            # +credit / -debit
            amount = credit - debit

            repo = repositories.BankStatementLineRepository(db)
            await line_repo.acreate(
                organization_id=group.organization_id,
                bank_statement_id=statement.pk,
                transaction_date=transaction_date,
                reference_number=reference_number,
                description=description.strip(),
                amount=amount,
            )

    elif bank_code == "BCA":
        # Extract header info from the first ~12 rows
        for i in range(len(df)):
            cell = str(df.iloc[i, 0]).upper()

            if "PERIODE" in cell:
                header["PERIOD"] = df.iloc[i, 0].split(":", 1)[1].strip()

            if "SALDO AWAL" in cell:
                header["OPENING_BALANCE"] = df.iloc[i, 0].split(":", 1)[1].strip()

            if "SALDO AKHIR" in cell:
                header["CLOSING_BALANCE"] = df.iloc[i, 0].split(":", 1)[1].strip()

            if "MUTASI DEBET" in cell:
                header["TOTAL_DEBIT"] = df.iloc[i, 0].split(":", 1)[1].strip()

            if "MUTASI KREDIT" in cell:
                header["TOTAL_CREDIT"] = df.iloc[i, 0].split(":", 1)[1].strip()

        # Validate essential fields exist
        required = ["PERIOD", "OPENING_BALANCE", "CLOSING_BALANCE"]
        for field in required:
            if field not in header:
                raise HTTPException(400, f"BCA: Missing required field: {field}")

        # Parse period
        try:
            start_date_str, end_date_str = header["PERIOD"].split("-")
            period_start = datetime.strptime(start_date_str.strip(), "%d/%m/%Y").date()
            period_end = datetime.strptime(end_date_str.strip(), "%d/%m/%Y").date()
        except Exception:
            raise HTTPException(status_code=400, detail="BCA: Invalid Periode format")

        opening_balance = clean_decimal(header["OPENING_BALANCE"])
        closing_balance = clean_decimal(header["CLOSING_BALANCE"])
        total_debit = clean_decimal(header.get("TOTAL_DEBIT", "0"))
        total_credit = clean_decimal(header.get("TOTAL_CREDIT", "0"))
        statement_date = period_end

        statement = await statement_repo.acreate(
            bank_account_id=bank_account_id,
            organization_id=group.organization_id,
            created_by_id=request_user.pk,
            statement_date=statement_date,
            opening_balance=opening_balance,
            closing_balance=closing_balance,
            period_start=period_start,
            period_end=period_end,
            total_debit=total_debit,
            total_credit=total_credit,
            status=enums.BankStatementStatus.IMPORTED,
            remark="Imported",
        )

        # Find transaction header row ("Tanggal Transaksi")
        txn_start_row = None
        for idx, row in df.iterrows():
            if str(row[0]).strip().upper().startswith("TANGGAL TRANSAKSI"):
                txn_start_row = idx + 1
                break
        if txn_start_row is None:
            raise HTTPException(400, "BCA: Transaction table not found")

        print("DEBUG TRANSACTION ROWS:")
        print(df.iloc[txn_start_row:txn_start_row+5].to_string())

        # Parse transaction lines
        for idx, row in df.iloc[txn_start_row:].iterrows():
            cell = str(row[0]).strip()
            # Only process if the first column is a date (skip summary rows)
            try:
                transaction_date = date_parse(cell, dayfirst=True).date()
            except Exception:
                continue  # Skip rows that are not transactions

            description = str(row[1]) if not pd.isna(row[1]) else ""
            amount_str = str(row[3]) if len(row) > 3 and not pd.isna(row[3]) and str(row[3]).strip() else "0"
            # Amount and direction
            if "CR" in amount_str:
                amount = clean_decimal(amount_str.replace("CR", ""))
            elif "DB" in amount_str:
                amount = -clean_decimal(amount_str.replace("DB", ""))
            else:
                amount = clean_decimal(amount_str)

            await line_repo.acreate(
                organization_id=group.organization_id,
                bank_statement_id=statement.pk,
                transaction_date=transaction_date,
                reference_number=None,
                description=description.strip(),
                amount=amount,
            )

    else:
        raise HTTPException(
            status_code=400,
            detail=f"Bank format not yet supported for bank_code={bank_code}. Please contact admin."
        )

    statement = await statement_repo.aget_or_404(statement.pk)
    return schemas.BankStatement.model_validate(statement)



# Bank Statement Line

async def create_bank_statement_line(
    db: AsyncSession,
    bank_statement_line: schemas.BankStatementLineCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankStatementLineRepository(db)
    obj = await repo.acreate(
        **bank_statement_line.model_dump(),
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
    )
    return obj

async def list_bank_statement_lines(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.BankStatementLineFilter,
    ordering: list[enums.BankStatementLineOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankStatementLineRepository(db, organization_id=group.organization_id)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def delete_bank_statement_lines(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankStatementLineRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)


async def auto_match_bank_reconcile(
    db: AsyncSession,
    bank_statement_id: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group

    # Get AR/AP account IDs for this organization
    from be_accounting.coa.models import AccountMapping
    mapping_stmt = select(AccountMapping).where(AccountMapping.organization_id == group.organization_id)
    mapping_result = await db.execute(mapping_stmt)
    account_mapping = mapping_result.scalar_one_or_none()
    if not account_mapping:
        raise HTTPException(400, "Account mapping not found for organization")

    ar_account_id = account_mapping.account_receivable_id
    ap_account_id = getattr(account_mapping, "account_payable_id", None)  # Add this field if not present

    line_repo = repositories.BankStatementLineRepository(db, organization_id=group.organization_id)
    bank_lines = await line_repo.alist_by_statement_id(bank_statement_id)

    matched_count = 0
    unmatched_count = 0

    for bank_line in bank_lines:
        if bank_line.is_reconciled:
            matched_count += 1
            continue

        # Determine side based on amount
        side = EntrySide.CREDIT if bank_line.amount > 0 else EntrySide.DEBIT

        # Find matching entry items for AR/AP accounts only
        stmt = (
            select(EntryItem)
            .where(
                EntryItem.amount == bank_line.amount,
                EntryItem.side == side,
                EntryItem.account_id.in_([ar_account_id, ap_account_id] if ap_account_id else [ar_account_id]),
            )
            .options(selectinload(EntryItem.account))
        )
        result = await db.execute(stmt)
        matches = result.scalars().all()

        # Only match if the account and side logic is correct
        filtered_matches = []
        for entry_item in matches:
            if entry_item.account_id == ar_account_id and side == EntrySide.CREDIT:
                filtered_matches.append(entry_item)
            elif ap_account_id and entry_item.account_id == ap_account_id and side == EntrySide.DEBIT:
                filtered_matches.append(entry_item)

        if filtered_matches:
            reconcile_repo = repositories.BankReconcileRepository(db, organization_id=group.organization_id)
            for entry_line in filtered_matches:
                await reconcile_repo.acreate(
                    bank_statement_line_id=bank_line.pk,
                    journal_entry_id=entry_line.entry_id,
                    matched_amount=bank_line.amount,
                    status=enums.BankReconStatus.DRAFT.value,
                    matched_at=datetime.utcnow(),
                    organization_id=group.organization_id,
                    created_by_id=request_user.pk,
                )
            await line_repo.aupdate(bank_line.pk, is_reconciled=True)
            matched_count += 1
        else:
            unmatched_count += 1

    return schemas.AutoMatchResult(
        matched=matched_count,
        remaining_unmatched=unmatched_count,
    )


async def manual_match_bank_reconcile(
    db: AsyncSession,
    bank_reconcile: schemas.BankReconcileCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group

    reconcile_repo = repositories.BankReconcileRepository(db, organization_id=group.organization_id)
    bank_line_repo = repositories.BankStatementLineRepository(db, organization_id=group.organization_id)
    journal_entry_repo = JournalEntryRepository(db, organization_id=group.organization_id)

    # Create Bank Reconcile record
    obj = await reconcile_repo.acreate(
        bank_statement_line_id=bank_reconcile.bank_statement_line_id,
        journal_entry_id=bank_reconcile.journal_entry_id,
        matched_amount=bank_reconcile.matched_amount,
        status=enums.BankReconStatus.DRAFT.value,
        matched_at=datetime.utcnow(),
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
    )

    # Update Bank Statement Line as reconciled
    await bank_line_repo.aupdate(bank_reconcile.bank_statement_line_id, is_reconciled=True)

    # --- Journal Entry Logic ---
    # Fetch account mapping
    pagination = PaginationQuery(page=1, size=1)
    mappings = await list_account_mappings(db, pagination, request_user)
    items = list(mappings["items"]) if isinstance(mappings, dict) else list(mappings.items)
    mapping = items[0] if items else None
    if not mapping:
        raise Exception("Account mapping not set.")

    entry_obj = await journal_entry_repo.aget_or_404(bank_reconcile.journal_entry_id)

    # Get correct account IDs for Outstanding Receipts and AR
    outstanding_receipts_account_id = mapping.outstanding_receipts_account_id
    outstanding_payments_account_id = mapping.outstanding_payments_account_id
    bank_account_id = mapping.bank_account_id


    entry_items = []
    if entry_obj.entry_type == journal_enums.EntryTypeEnum.SALES:
        entry_items = [
            # Bank Account (Dr)
            {
                "account_id": bank_account_id,
                "amount": obj.amount,
                "side": journal_enums.EntrySide.DEBIT,
                "description": f"Payment {obj.pk} Bank Account"
            },
            # Outstanding Receipts (Cr)
            {
                "account_id": outstanding_receipts_account_id,
                "amount": obj.amount,
                "side": journal_enums.EntrySide.CREDIT,
                "description": f"Payment {obj.pk} Outstanding Receipts"
            },
        ]
    elif entry_obj.entry_type == journal_enums.EntryTypeEnum.PURCHASE:
        entry_items = [
            # Outstanding Payments (Dr)
            {
                "account_id": outstanding_payments_account_id,
                "amount": obj.amount,
                "side": journal_enums.EntrySide.DEBIT,
                "description": f"Payment {obj.pk} Outstanding Payments"
            },
            # Bank Account (Cr)
            {
                "account_id": bank_account_id,
                "amount": obj.amount,
                "side": journal_enums.EntrySide.CREDIT,
                "description": f"Payment {obj.pk} Bank Account"
            },
        ]

    entry = journal_schemas.EntryCreate(
        date=obj.payment_date,
        description=f"Customer Payment {obj.pk} for Invoice {obj.invoice.pk}",
        entry_type=journal_enums.EntryTypeEnum.BANK,
        items=entry_items,
        currency_id=obj.currency_id,
    )
    journal_entry = await journal_utils.record_entry(db, entry, request_user)

    await bank_line_repo.aupdate(bank_reconcile.bank_statement_line_id, journal_entry_id=journal_entry.pk)

    return obj


async def approve_bank_reconcile(
    db: AsyncSession,
    bank_reconcile_id: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group

    reconcile_repo = repositories.BankReconcileRepository(db, organization_id=group.organization_id)
    bank_line_repo = repositories.BankStatementLineRepository(db, organization_id=group.organization_id)

    reconcile = await reconcile_repo.aget_or_404(bank_reconcile_id)

    if reconcile.status == enums.BankReconStatus.VALIDATED:
        raise HTTPException(400, "Bank Reconcile already approved")

    # Update status to APPROVED
    await reconcile_repo.aupdate(
        bank_reconcile_id,
        status=enums.BankReconStatus.VALIDATED,
        last_modified_by_id=request_user.pk,
    )

    # Additional logic such as creating journal entries can be added here

    return reconcile
