from datetime import date
from typing import List, Optional
from fastapi import File, UploadFile
from typing_extensions import Annotated

from pydantic import PositiveInt, StringConstraints

from be_kit.paginations import BasePaginatedResponse
from be_kit.schemas import BaseSchema, BaseORMSchema, Decimal
from be_core_services.country.schemas import BankMin, CurrencyMin
from be_uam.kit.schemas import MetadataMixin
from be_uam.user.schemas import UserMin
from be_accounting.bank_account.schemas import BankAccount
from be_accounting.journal.schemas import EntryMin
from ..coa.schemas import AccountMin
from . import enums


# Bank Statement Line

class BankStatementLineBase(BaseSchema):
    transaction_date: date
    description: Optional[str] = None
    reference_number: Optional[str] = None
    amount: Decimal

class BankStatementLineCreate(BankStatementLineBase):
    pass

class BankStatementLineUpdate(BankStatementLineBase):
    is_reconciled: Optional[bool] = None
    # journal_entry_id: Optional[int] = None

class BankStatementLine(BankStatementLineBase, BaseORMSchema):
    is_reconciled: bool
    # journal_entry_id: Optional[int] = None

class PaginatedBankStatementLine(BasePaginatedResponse):
    items: list[BankStatementLine]

class BankStatementLineFilter(BaseSchema):
    bank_statement_id: PositiveInt | None = None
    bank_statement_ids: Optional[List[int]] = None
    transaction_date: date | None = None
    amount: Decimal | None = None

# Bank Statement

class BaseBankStatement:
    statement_date: Annotated[date, None]
    opening_balance: Annotated[Decimal | None, None]
    closing_balance: Annotated[Decimal | None, None]
    remark: Annotated[str | None, StringConstraints(max_length=500)]


class BankStatementCreate(BaseBankStatement, BaseSchema):
    bank_account_id: PositiveInt
    lines: Optional[List[BankStatementLineCreate]] = None


class BankStatementUpdate(BaseBankStatement, BaseSchema):
    bank_account_id: PositiveInt
    lines: Optional[List[BankStatementLineUpdate]] = None


class BankStatement(MetadataMixin, BaseBankStatement, BaseORMSchema):
    bank_account: BankAccount
    lines: List[BankStatementLine]


class PaginatedBankStatement(BasePaginatedResponse):
    items: list[BankStatement]


class BankStatementFilter(BaseSchema):
    statement_date: date | None = None
    opening_balance: Decimal | None = None
    opening_balance_icontains: Decimal | None = None
    closing_balance: Decimal | None = None
    closing_balance_icontains: Decimal | None = None
    bank_account_id: PositiveInt | None = None


class BankStatementMin(BaseBankStatement, BaseORMSchema):
    pass


class BankStatementOpt(BankStatementMin):
    pass


class PaginatedBankStatementOpt(BasePaginatedResponse):
    items: list[BankStatementOpt]


class BankStatementUpload(BaseSchema):
    file: Annotated[UploadFile, File()]



class BaseBankReconcile(BaseSchema):
    matched_amount: Decimal | None = None
    status: enums.BankReconStatus | None = None
    matched_at: Optional[date] = None


class BankReconcile(BaseBankReconcile, MetadataMixin, BaseORMSchema):
    journal_entry: EntryMin
    bank_statement_line: BankStatementLine

class PaginatedBankReconcile(BasePaginatedResponse):
    items: list[BankReconcile]

class AutoMatchResult(BaseSchema):
    matched: int
    remaining_unmatched: int

class BankReconcileCreate(BaseSchema):
    bank_statement_line_id: PositiveInt
    journal_entry_id: PositiveInt

UserMin.model_rebuild()
EntryMin.model_rebuild()
BankStatementMin.model_rebuild()
