from datetime import date, datetime
from decimal import Decimal

from sqlalchemy import (
    BigInteger,
    Date,
    DateTime,
    Enum,
    ForeignKey,
    String,
    UniqueConstraint,
)
from sqlalchemy.orm import Mapped, declared_attr, mapped_column, relationship

from be_kit.databases import BaseModel, SoftDeleteMixin, NominalField
from be_uam.kit.models import MetadataMixin, OrganizationMixin
from be_uam.user.models import User
from be_core_services.country.models import Bank, Currency
from be_core_services.tag.models import Tag

from be_accounting.bank_account.models import BankAccount
from be_accounting.journal.models import Entry
from ..coa.models import Account
from . import enums


class BankStatement(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "accounting_bank_statement"

    bank_account_id: Mapped[int] = mapped_column(
        ForeignKey("accounting_bank_account.pk", ondelete="CASCADE"),
    )
    bank_account: Mapped["BankAccount"] = relationship(
        "BankAccount",
        passive_deletes=True,
        lazy="selectin",
    )
    statement_date: Mapped[date] = mapped_column(Date, nullable=False)
    opening_balance: Mapped[Decimal] = mapped_column(NominalField, nullable=True, default=0)
    closing_balance: Mapped[Decimal] = mapped_column(NominalField, nullable=True, default=0)
    period_start: Mapped[date] = mapped_column(Date, nullable=False)
    period_end: Mapped[date] = mapped_column(Date, nullable=False)
    total_debit: Mapped[int] = mapped_column(nullable=True, default=0)
    total_credit: Mapped[int] = mapped_column(nullable=True, default=0)
    remark: Mapped[str | None] = mapped_column(String(500), nullable=True)
    status: Mapped[enums.BankStatementStatus] = mapped_column(
        Enum(enums.BankStatementStatus),
        index=True,
        default=enums.BankStatementStatus.DRAFT,
        nullable=True,
    )
    lines: Mapped[list["BankStatementLine"]] = relationship(
        "BankStatementLine",
        back_populates="bank_statement",
        cascade="all, delete-orphan",
        lazy="selectin",
    )

class BankStatementLine(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "accounting_bank_statement_line"

    bank_statement_id: Mapped[int] = mapped_column(
        ForeignKey("accounting_bank_statement.pk", ondelete="CASCADE")
    )

    transaction_date: Mapped[date] = mapped_column(Date, nullable=False)
    description: Mapped[str | None] = mapped_column(String(500), nullable=True)
    reference_number: Mapped[str | None] = mapped_column(String(100), nullable=True)
    amount: Mapped[Decimal] = mapped_column(NominalField, nullable=False)  # +credit / -debit

    is_reconciled: Mapped[bool] = mapped_column(default=False)

    bank_statement: Mapped[BankStatement | None] = relationship(
        "BankStatement",
        foreign_keys=[bank_statement_id],
        back_populates="lines",
        passive_deletes=True,
        lazy="selectin",
    )


class BankReconcile(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "accounting_bank_reconcile"

    bank_statement_line_id: Mapped[int] = mapped_column(
        ForeignKey("accounting_bank_statement_line.pk", ondelete="CASCADE")
    )
    journal_entry_id: Mapped[int | None] = mapped_column(
        BigInteger, ForeignKey(Entry.pk, ondelete="SET NULL"), nullable=True
    )

    bank_statement_line: Mapped[BankStatementLine | None] = relationship(
        "BankStatementLine",
        foreign_keys=[bank_statement_line_id],
        passive_deletes=True,
        lazy="selectin",
    )
    journal_entry: Mapped["Entry | None"] = relationship(
        Entry,
        foreign_keys=[journal_entry_id],
    )

    matched_amount: Mapped[Decimal] = mapped_column(NominalField, nullable=False)
    status: Mapped[enums.BankReconStatus] = mapped_column(
        Enum(enums.BankReconStatus),
        index=True,
        default=enums.BankReconStatus.DRAFT,
        nullable=True,
    )
    matched_at: Mapped[datetime] = mapped_column(DateTime, default=datetime.utcnow)
