from typing import Annotated

from fastapi import APIRouter, Depends, Form, Query, status
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user, authorize
from ..databases import get_async_session
from . import enums, schemas, utils
from .permissions import bank_statement_permission_mapper


bank_statement_router = APIRouter(tags=["bank_statement"], prefix="/bank_statement")


@bank_statement_router.post(
    "/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["create_bank_statement"]))],
    response_model=schemas.BankStatement,
    status_code=status.HTTP_201_CREATED,
)
async def create_bank_statement(
    bank_statement: schemas.BankStatementCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_bank_statement(session, bank_statement, request_user)


@bank_statement_router.get(
    "/{bank_statement_id:int}/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["retrieve_bank_statement"]))],
    response_model=schemas.BankStatement,
)
async def retrieve_bank_statement(
    bank_statement_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_bank_statement(session, bank_statement_id, request_user)


@bank_statement_router.get(
    "/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["list_bank_statement"]))],
    response_model=schemas.PaginatedBankStatement,
)
async def list_bank_statement(
    pagination: PaginationQuery = Depends(),
    filters: schemas.BankStatementFilter = Depends(),
    ordering: Annotated[list[enums.BankStatementOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_bank_statements(
        session, pagination, filters, ordering, request_user
    )


@bank_statement_router.get(
    "/options/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["list_bank_statement_option"]))],
    response_model=schemas.PaginatedBankStatementOpt,
)
async def list_bank_statement_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.BankStatementFilter = Depends(),
    ordering: Annotated[list[enums.BankStatementOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_bank_statements(
        session, pagination, filters, ordering, request_user
    )


@bank_statement_router.put(
    "/{bank_statement_id:int}/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["update_bank_statement"]))],
    response_model=schemas.BankStatement,
)
async def update_bank_statement(
    bank_statement: schemas.BankStatementUpdate,
    bank_statement_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_bank_statement(
        session, bank_statement_id, bank_statement, request_user
    )


@bank_statement_router.delete(
    "/{bank_statement_id:int}/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["delete_bank_statement"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_bank_statement(
    bank_statement_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_bank_statement(session, bank_statement_id, request_user)


@bank_statement_router.get(
    "/{bank_account_id:int}/list/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["list_bank_statement"]))],
    response_model=schemas.PaginatedBankStatement,
)
async def list_bank_statement_by_bank_id(
    bank_account_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
    pagination: PaginationQuery = Depends(),
):
    return await utils.list_bank_statements_by_bank_id(
        session, bank_account_id, request_user, pagination
    )

@bank_statement_router.get(
    "/{bank_statement_id:int}/lines/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["list_bank_statement"]))],
    response_model=schemas.PaginatedBankStatementLine,
)
async def list_bank_statement_line_by_statement(
    bank_statement_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
    pagination: PaginationQuery = Depends(),
):
    return await utils.list_bank_statement_line_by_statement(
        session, bank_statement_id, request_user, pagination
    )

@bank_statement_router.get(
    "/{bank_account_id:int}/statement_lines/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["list_bank_statement"]))],
    response_model=schemas.PaginatedBankStatementLine,
)
async def list_bank_statement_line_by_bank_id(
    bank_account_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
    pagination: PaginationQuery = Depends(),
):
    return await utils.list_bank_statement_line_by_bank_id(
        session, bank_account_id, request_user, pagination
    )


@bank_statement_router.post(
    "/{bank_account_id:int}/upload/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["upload_bank_statement"]))],
    status_code=status.HTTP_201_CREATED,
)
async def upload_bank_statement(
    bank_statement_upload: Annotated[
        schemas.BankStatementUpload, Form(media_type="multipart/form-data")
    ],
    bank_account_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.handle_bank_statement_upload(session, bank_statement_upload, bank_account_id, request_user)


bank_statement_line_router = APIRouter(tags=["bank_statement_line"], prefix="/bank_statement_line")


@bank_statement_line_router.post(
    "/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["create_bank_statement"]))],
    response_model=schemas.BankStatementLine,
    status_code=status.HTTP_201_CREATED,
)
async def create_bank_statement_line(
    bank_statement: schemas.BankStatementLineCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_bank_statement(session, bank_statement, request_user)


@bank_statement_line_router.get(
    "/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["list_bank_statement"]))],
    response_model=schemas.PaginatedBankStatementLine,
)
async def list_bank_statement_line(
    pagination: PaginationQuery = Depends(),
    filters: schemas.BankStatementLineFilter = Depends(),
    ordering: Annotated[list[enums.BankStatementLineOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_bank_statement_lines(
        session, pagination, filters, ordering, request_user
    )


@bank_statement_line_router.delete(
    "/{bank_statement_line_id:int}/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["delete_bank_statement"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_bank_statement_line(
    bank_statement_line_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_bank_statement_lines(session, bank_statement_line_id, request_user)


bank_reconcile_router = APIRouter(tags=["bank_reconcile"], prefix="/bank_reconcile")


@bank_reconcile_router.post(
    "/statement/{bank_statement_id:int}/auto-match/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["create_bank_statement"]))],
    response_model=schemas.AutoMatchResult,
    status_code=status.HTTP_200_OK,
)
async def auto_match_bank_reconcile(
    bank_statement_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.auto_match_bank_reconcile(session, bank_statement_id, request_user)

@bank_reconcile_router.post(
    "/statement_line/{bank_statement_line_id:int}/match/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["create_bank_statement"]))],
    response_model=schemas.BankReconcile,
    status_code=status.HTTP_201_CREATED,
)
async def manual_match_bank_reconcile(
    bank_reconcile: schemas.BankReconcileCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.manual_match_bank_reconcile(session, bank_reconcile, request_user)

@bank_reconcile_router.post(
    "/{bank_reconcile_id:int}/approve/",
    dependencies=[Depends(authorize(bank_statement_permission_mapper["approve_bank_reconcile"]))],
    response_model=schemas.BankReconcile,
)
async def approve_bank_reconcile(
    bank_reconcile_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.approve_bank_reconcile(session, bank_reconcile_id, request_user)
