import io

import pandas as pd
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from . import enums, schemas, repositories


async def create_bank_account(
    db: AsyncSession,
    bank_account: schemas.BankAccountCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankAccountRepository(db)
    obj = await repo.acreate(
        **bank_account.model_dump(),
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
    )
    return obj


async def retrieve_bank_account(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankAccountRepository(db, organization_id=group.organization_id)
    obj = await repo.aget_or_404(pk)
    return obj


async def list_bank_accounts(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.BankAccountFilter,
    ordering: list[enums.BankAccountOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankAccountRepository(db, organization_id=group.organization_id)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def update_bank_account(
    db: AsyncSession,
    pk: int,
    bank_account: schemas.BankAccountUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankAccountRepository(db, organization_id=group.organization_id)
    obj = await repo.aupdate(
        pk, **bank_account.model_dump(), last_modified_by_id=request_user.pk
    )
    return obj


async def delete_bank_account(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.BankAccountRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)
