from typing_extensions import Annotated

from pydantic import PositiveInt, StringConstraints

from be_kit.paginations import BasePaginatedResponse
from be_kit.schemas import BaseSchema, BaseORMSchema
from be_core_services.country.schemas import BankMin, CurrencyMin
from be_uam.kit.schemas import MetadataMixin
from be_uam.user.schemas import UserMin
from ..coa.schemas import AccountMin
from . import enums


class BaseBankAccount:
    account_number: Annotated[str, StringConstraints(min_length=1, max_length=20)]
    name: Annotated[str, StringConstraints(min_length=1, max_length=100)]


class BankAccountCreate(BaseBankAccount, BaseSchema):
    bank_id: PositiveInt
    currency_id: PositiveInt
    account_id: PositiveInt | None


class BankAccountUpdate(BaseBankAccount, BaseSchema):
    bank_id: PositiveInt
    currency_id: PositiveInt
    account_id: PositiveInt | None


class BankAccount(MetadataMixin, BaseBankAccount, BaseORMSchema):
    bank: BankMin
    currency: CurrencyMin
    account: AccountMin | None


class PaginatedBankAccount(BasePaginatedResponse):
    items: list[BankAccount]


class BankAccountFilter(BaseSchema):
    name: str | None = None
    name__icontains: str | None = None
    account_number: str | None = None
    account_number__icontains: str | None = None
    bank_id: PositiveInt | None = None
    currency_id: PositiveInt | None = None
    account_id: PositiveInt | None = None


class BankAccountMin(BaseBankAccount, BaseORMSchema):
    pass


class BankAccountOpt(BankAccountMin):
    pass


class PaginatedBankAccountOpt(BasePaginatedResponse):
    items: list[BankAccountOpt]


UserMin.model_rebuild()
