from typing import Annotated

from fastapi import APIRouter, Depends, Query, status
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user, authorize
from ..databases import get_async_session
from . import enums, schemas, utils
from .permissions import permission_mapper


router = APIRouter(tags=["bank_accounts"], prefix="/bank_account")


@router.post(
    "/",
    dependencies=[Depends(authorize(permission_mapper["create_bank_account"]))],
    response_model=schemas.BankAccount,
    status_code=status.HTTP_201_CREATED,
)
async def create_bank_account(
    bank_account: schemas.BankAccountCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_bank_account(session, bank_account, request_user)


@router.get(
    "/{bank_account_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["retrieve_bank_account"]))],
    response_model=schemas.BankAccount,
)
async def retrieve_bank_account(
    bank_account_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_bank_account(session, bank_account_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(permission_mapper["list_bank_account"]))],
    response_model=schemas.PaginatedBankAccount,
)
async def list_bank_account(
    pagination: PaginationQuery = Depends(),
    filters: schemas.BankAccountFilter = Depends(),
    ordering: Annotated[list[enums.BankAccountOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_bank_accounts(
        session, pagination, filters, ordering, request_user
    )


@router.get(
    "/options/",
    dependencies=[Depends(authorize(permission_mapper["list_bank_account_option"]))],
    response_model=schemas.PaginatedBankAccountOpt,
)
async def list_bank_account_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.BankAccountFilter = Depends(),
    ordering: Annotated[list[enums.BankAccountOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_bank_accounts(
        session, pagination, filters, ordering, request_user
    )


@router.put(
    "/{bank_account_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["update_bank_account"]))],
    response_model=schemas.BankAccount,
)
async def update_bank_account(
    bank_account: schemas.BankAccountUpdate,
    bank_account_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_bank_account(
        session, bank_account_id, bank_account, request_user
    )


@router.delete(
    "/{bank_account_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["delete_bank_account"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_bank_account(
    bank_account_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_bank_account(session, bank_account_id, request_user)
