import io

import pandas as pd
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from . import enums, schemas, repositories


async def create_sales_team(
    db: AsyncSession,
    sales_team: schemas.SalesTeamCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.SalesTeamRepository(db)
    member_repo = repositories.MemberRepository(db)
    sales_team_data = sales_team.model_dump()
    members = sales_team_data.pop("members")
    obj = await repo.acreate(
        **sales_team_data,
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
        _commit=False,
    )
    for member in members:
        await member_repo.acreate(**member, sales_team_id=obj.pk)

    await db.commit()
    await db.refresh(obj)
    return obj


async def retrieve_sales_team(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.SalesTeamRepository(db, organization_id=group.organization_id)
    obj = await repo.aget_or_404(pk)
    return obj


async def list_sales_teams(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.SalesTeamFilter,
    ordering: list[enums.SalesTeamOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.SalesTeamRepository(db, organization_id=group.organization_id)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def update_sales_team(
    db: AsyncSession,
    pk: int,
    sales_team: schemas.SalesTeamUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.SalesTeamRepository(db, organization_id=group.organization_id)
    member_repo = repositories.MemberRepository(db)
    sales_team_data = sales_team.model_dump()
    members = sales_team_data.pop("members")
    obj = await repo.aupdate(pk, **sales_team_data, last_modified_by_id=request_user.pk)

    existing_members = await member_repo.arecords({"sales_team_id": obj.pk})
    for existing in existing_members:
        await member_repo.adelete(existing.pk)

    for member in members:
        await member_repo.acreate(**member.model_dump(), sales_team_id=obj.pk)

    await db.commit()
    await db.refresh(obj)
    return obj


async def delete_sales_team(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.SalesTeamRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)


async def get_sales_team_by_user(
    db: AsyncSession,
    user_id: int,
    organization_id: int,
):
    repo = repositories.SalesTeamRepository(db, organization_id=organization_id)
    obj = await repo.aget_by_user(user_id)
    return obj
