from sqlalchemy import (
    BigInteger,
    Enum,
    ForeignKey,
    String,
    UniqueConstraint,
)
from sqlalchemy.orm import Mapped, declared_attr, mapped_column, relationship

from be_kit.databases import BaseModel, SoftDeleteMixin
from be_uam.kit.models import MetadataMixin, OrganizationMixin
from be_uam.user.models import User
from be_core_services.tag.models import Tag
from . import enums


class SalesTeam(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "crm_sales_team"
    __table_args__ = (UniqueConstraint("organization_id", "name"),)

    name: Mapped[str] = mapped_column(String(100))
    description: Mapped[str | None] = mapped_column(String(500), nullable=True)
    expected_revenue: Mapped[str | None] = mapped_column(String(100), nullable=True)
    leader_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey(f"{User.__tablename__}.pk", ondelete="CASCADE"),
    )
    leader: Mapped[User] = relationship(
        User, foreign_keys=[leader_id], passive_deletes=True, lazy="selectin"
    )
    members: Mapped[list["Member"]] = relationship(
        back_populates="sales_team", cascade="all, delete-orphan", lazy="selectin"
    )
    tags: Mapped[list[Tag]] = relationship(
        secondary="crm_sales_team_tag", lazy="selectin"
    )


class Member(BaseModel):
    __tablename__ = "crm_sales_team_member"
    __table_args__ = (UniqueConstraint("sales_team_id", "user_id"),)

    sales_team_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey(f"{SalesTeam.__tablename__}.pk", ondelete="CASCADE"),
        primary_key=True,
    )
    sales_team: Mapped["SalesTeam"] = relationship(
        SalesTeam, back_populates="members", passive_deletes=True, lazy="selectin"
    )
    user_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey(f"{User.__tablename__}.pk", ondelete="CASCADE"),
    )
    user: Mapped[User] = relationship(User, passive_deletes=True, lazy="selectin")


class SalesTeamTag(BaseModel):
    __tablename__ = "crm_sales_team_tag"
    __table_args__ = (UniqueConstraint("sales_team_id", "tag_id"),)

    sales_team_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey("crm_sales_team.pk", ondelete="CASCADE"),
        index=True,
    )
    tag_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey(f"{Tag.__tablename__}.pk", ondelete="CASCADE"),
        index=True,
    )
