from typing import Annotated

from fastapi import APIRouter, Depends, Query, status
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user, authorize
from ..databases import get_async_session
from . import enums, schemas, utils
from .permissions import permission_mapper


router = APIRouter(tags=["sales_teams"], prefix="/sales_team")


@router.post(
    "/",
    dependencies=[Depends(authorize(permission_mapper["create_sales_team"]))],
    response_model=schemas.SalesTeam,
    status_code=status.HTTP_201_CREATED,
)
async def create_sales_team(
    sales_team: schemas.SalesTeamCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_sales_team(session, sales_team, request_user)


@router.get(
    "/{sales_team_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["retrieve_sales_team"]))],
    response_model=schemas.SalesTeam,
)
async def retrieve_sales_team(
    sales_team_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_sales_team(session, sales_team_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(permission_mapper["list_sales_team"]))],
    response_model=schemas.PaginatedSalesTeam,
)
async def list_sales_team(
    pagination: PaginationQuery = Depends(),
    filters: schemas.SalesTeamFilter = Depends(),
    ordering: Annotated[list[enums.SalesTeamOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_sales_teams(
        session, pagination, filters, ordering, request_user
    )


@router.get(
    "/options/",
    dependencies=[Depends(authorize(permission_mapper["list_sales_team_option"]))],
    response_model=schemas.PaginatedSalesTeamOpt,
)
async def list_sales_team_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.SalesTeamFilter = Depends(),
    ordering: Annotated[list[enums.SalesTeamOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_sales_teams(
        session, pagination, filters, ordering, request_user
    )


@router.put(
    "/{sales_team_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["update_sales_team"]))],
    response_model=schemas.SalesTeam,
)
async def update_sales_team(
    sales_team: schemas.SalesTeamUpdate,
    sales_team_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_sales_team(
        session, sales_team_id, sales_team, request_user
    )


@router.delete(
    "/{sales_team_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["delete_sales_team"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_sales_team(
    sales_team_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_sales_team(session, sales_team_id, request_user)


@router.get(
    "/by_user/{user_id:int}/",
    response_model=schemas.SalesTeamMin,
)
async def get_sales_team_by_user(
    user_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    group = await request_user.awaitable_attrs.group
    return await utils.get_sales_team_by_user(session, user_id, group.organization_id)
