import io

from be_uam.user.repositories import UserRepository
import pandas as pd
from pydantic import ValidationError
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from ..sales_team.repositories import SalesTeamRepository
from ..customer.repositories import CustomerRepository
from . import enums, schemas, repositories


async def create_lead(
    db: AsyncSession,
    lead: schemas.LeadCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.LeadRepository(db)
    obj = await repo.acreate(
        **lead.model_dump(),
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
    )
    return obj


async def retrieve_lead(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.LeadRepository(db, organization_id=group.organization_id)
    obj = await repo.aget_or_404(pk)
    return obj


async def list_leads(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.LeadFilter,
    ordering: list[enums.LeadOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.LeadRepository(db, organization_id=group.organization_id)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def update_lead(
    db: AsyncSession,
    pk: int,
    lead: schemas.LeadUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.LeadRepository(db, organization_id=group.organization_id)
    obj = await repo.aupdate(
        pk, **lead.model_dump(), last_modified_by_id=request_user.pk
    )
    return obj


async def delete_lead(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.LeadRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)


async def handle_lead_upload(
    db: AsyncSession, lead_upload: schemas.LeadUpload, request_user: User
):
    buffer = await lead_upload.file.read()
    lead_data = pd.read_excel(buffer, sheet_name="lead")

    if lead_data is None or "customer_name" not in lead_data.columns:
        raise ValueError("Invalid template format. Missing 'customer_name' column.")

    group = await request_user.awaitable_attrs.group
    customer_repo = CustomerRepository(db, organization_id=group.organization_id)
    lead_repo = repositories.LeadRepository(db, organization_id=group.organization_id)

    for _, row in lead_data.iterrows():
        customer_name = row["customer_name"]
        customers = await customer_repo.arecords({"contact___name": customer_name})
        if len(customers) == 0:
            raise ValidationError(f"Customer with name '{customer_name}' not found.")
        sales_team_name = row["sales_team_name"]
        sales_team_repo = SalesTeamRepository(db, organization_id=group.organization_id)
        sales_teams = await sales_team_repo.arecords({"name": sales_team_name})
        if len(sales_teams) == 0:
            raise ValidationError(
                f"Sales team with name '{sales_team_name}' not found."
            )

        lead_create = schemas.LeadCreate(
            name=row.get("name"),
            description=row.get("description"),
            source=row.get("source"),
            score=row.get("score"),
            expected_revenue=row.get("expected_revenue"),
            customer_id=customers[0].pk,
            sales_team_id=sales_teams[0].pk,
        )
        await lead_repo.acreate(
            **lead_create.model_dump(),
            organization_id=group.organization_id,
            created_by_id=request_user.pk,
            _commit=False,
        )

    await db.commit()


async def handle_lead_download(
    db: AsyncSession,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.LeadRepository(db, organization_id=group.organization_id)
    leads = await repo.arecords()

    lead_rows = []
    for lead in leads:
        customer = await lead.awaitable_attrs.customer
        contact = await customer.awaitable_attrs.contact if customer else None
        sales_team = await lead.awaitable_attrs.sales_team
        sales_person = await lead.awaitable_attrs.sales_person
        lead_rows.append(
            {
                "customer_name": getattr(contact, "name", None),
                "sales_team_name": getattr(sales_team, "name", None),
                "sales_person_name": getattr(sales_person, "name", None),
                "name": getattr(lead, "name", None),
                "description": getattr(lead, "description", None),
                "source": getattr(lead, "source", None),
                "score": getattr(lead, "score", None),
                "expected_revenue": getattr(lead, "expected_revenue", None),
            }
        )

    lead_df = pd.DataFrame(lead_rows)

    output = io.BytesIO()
    with pd.ExcelWriter(output, engine="xlsxwriter") as writer:
        lead_df.to_excel(writer, sheet_name="lead", index=False)
    output.seek(0)
    return output
