from typing_extensions import Annotated

from fastapi import File, UploadFile
from pydantic import PositiveInt, StringConstraints

from be_kit.paginations import BasePaginatedResponse
from be_kit.schemas import BaseSchema, BaseORMSchema, NominalType
from be_uam.kit.schemas import MetadataMixin
from be_uam.user.schemas import UserMin
from be_core_services.tag.schemas import TagListMixin, TagListWriteMixin
from ..customer.schemas import CustomerMin
from ..sales_team.schemas import SalesTeamMin
from . import enums


class BaseLead:
    status: enums.LeadStatus
    name: Annotated[str, StringConstraints(max_length=100)]
    description: Annotated[str | None, StringConstraints(max_length=500)]
    source: enums.LeadSource
    score: "NominalType | None"
    expected_revenue: "NominalType | None"


class LeadCreate(TagListWriteMixin, BaseLead, BaseSchema):
    sales_team_id: PositiveInt
    sales_person_id: PositiveInt
    customer_id: PositiveInt


class LeadUpdate(TagListWriteMixin, BaseLead, BaseSchema):
    sales_team_id: PositiveInt
    sales_person_id: PositiveInt
    customer_id: PositiveInt


class Lead(TagListMixin, MetadataMixin, BaseLead, BaseORMSchema):
    identifier: Annotated[str, StringConstraints(max_length=10)]
    customer: CustomerMin
    sales_team: SalesTeamMin
    sales_person: UserMin


class PaginatedLead(BasePaginatedResponse):
    items: list[Lead]


class LeadFilter(BaseSchema):
    identifier: str | None = None
    identifier__icontains: str | None = None
    name: str | None = None
    name__icontains: str | None = None
    description: str | None = None
    description__icontains: str | None = None
    status: enums.LeadStatus | None = None
    sales_team_id: PositiveInt | None = None
    customer_id: PositiveInt | None = None


class LeadMin(BaseLead, BaseORMSchema):
    identifier: Annotated[str, StringConstraints(max_length=10)]
    pass


class LeadOpt(LeadMin):
    customer: CustomerMin
    pass


class PaginatedLeadOpt(BasePaginatedResponse):
    items: list[LeadOpt]


class LeadUpload(BaseSchema):
    file: Annotated[UploadFile, File()]


class LeadValidator(BaseSchema):
    customer_name: Annotated[str, StringConstraints(min_length=1, max_length=100)]
    sales_team_name: Annotated[str, StringConstraints(min_length=1, max_length=100)]
    sales_person_name: Annotated[str, StringConstraints(min_length=1, max_length=100)]
    name: Annotated[str, StringConstraints(max_length=100)]
    description: Annotated[str | None, StringConstraints(max_length=500)]
    source: enums.LeadSource
    score: "NominalType | None"
    expected_revenue: "NominalType | None"


UserMin.model_rebuild()
