from datetime import date
from decimal import Decimal

from sqlalchemy import (
    BigInteger,
    Date,
    Enum,
    ForeignKey,
    String,
    UniqueConstraint,
)
from sqlalchemy.orm import Mapped, declared_attr, mapped_column, relationship

from be_kit.databases import BaseModel, SoftDeleteMixin, NominalField
from be_uam.kit.models import MetadataMixin, OrganizationMixin
from be_uam.user.models import User
from be_core_services.tag.models import Tag

from be_crm import sales_team
from ..customer.models import Customer
from ..sales_team.models import SalesTeam
from . import enums


class Lead(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "crm_lead"
    __table_args__ = (UniqueConstraint("organization_id", "name"),)

    identifier: Mapped[str] = mapped_column(String(10), unique=True, index=True)
    name: Mapped[str] = mapped_column(String(100))
    description: Mapped[str | None] = mapped_column(String(500), nullable=True)
    status: Mapped[enums.LeadStatus] = mapped_column(
        Enum(enums.LeadStatus), default=enums.LeadStatus.NEW, index=True
    )
    source: Mapped[enums.LeadSource] = mapped_column(Enum(enums.LeadSource), index=True)
    score: Mapped[Decimal | None] = mapped_column(NominalField, nullable=True)
    expected_revenue: Mapped[Decimal | None] = mapped_column(
        NominalField, nullable=True
    )
    customer_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey("crm_customer.pk", ondelete="CASCADE"),
    )
    customer: Mapped[Customer] = relationship(
        Customer, passive_deletes=True, lazy="selectin"
    )
    sales_team_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey(f"{SalesTeam.__tablename__}.pk", ondelete="CASCADE"),
    )
    sales_person_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey(f"{User.__tablename__}.pk", ondelete="RESTRICT"),
    )
    sales_team: Mapped[SalesTeam] = relationship(
        SalesTeam, passive_deletes=True, lazy="selectin"
    )
    sales_person: Mapped["User"] = relationship(
        "User", foreign_keys=[sales_person_id], lazy="joined",
    )
    tags: Mapped[list[Tag]] = relationship(secondary="crm_lead_tag", lazy="selectin")


class Counter(OrganizationMixin, BaseModel):
    __tablename__ = "crm_lead_counter"
    __table_args__ = (UniqueConstraint("organization_id", "counter_date"),)

    counter_date: Mapped[date] = mapped_column(Date, index=True)
    counter: Mapped[int] = mapped_column(BigInteger, default=0)


class LeadTag(BaseModel):
    __tablename__ = "crm_lead_tag"
    __table_args__ = (UniqueConstraint("lead_id", "tag_id"),)

    lead_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey("crm_lead.pk", ondelete="CASCADE"),
        index=True,
    )
    tag_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey(f"{Tag.__tablename__}.pk", ondelete="CASCADE"),
        index=True,
    )
