import io
from typing import Annotated

from isort import file
import pandas as pd
from fastapi import APIRouter, Depends, Form, Query, status
from fastapi.responses import StreamingResponse
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_kit.exim.utils import get_version_upload_template
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user, authorize
from ..databases import get_async_session
from . import enums, schemas, utils
from .permissions import permission_mapper


router = APIRouter(tags=["leads"], prefix="/lead")


@router.post(
    "/",
    dependencies=[Depends(authorize(permission_mapper["create_lead"]))],
    response_model=schemas.Lead,
    status_code=status.HTTP_201_CREATED,
)
async def create_lead(
    lead: schemas.LeadCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_lead(session, lead, request_user)


@router.get(
    "/{lead_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["retrieve_lead"]))],
    response_model=schemas.Lead,
)
async def retrieve_lead(
    lead_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_lead(session, lead_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(permission_mapper["list_lead"]))],
    response_model=schemas.PaginatedLead,
)
async def list_lead(
    pagination: PaginationQuery = Depends(),
    filters: schemas.LeadFilter = Depends(),
    ordering: Annotated[list[enums.LeadOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_leads(session, pagination, filters, ordering, request_user)


@router.get(
    "/options/",
    dependencies=[Depends(authorize(permission_mapper["list_lead_option"]))],
    response_model=schemas.PaginatedLeadOpt,
)
async def list_lead_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.LeadFilter = Depends(),
    ordering: Annotated[list[enums.LeadOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_leads(session, pagination, filters, ordering, request_user)


@router.put(
    "/{lead_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["update_lead"]))],
    response_model=schemas.Lead,
)
async def update_lead(
    lead: schemas.LeadUpdate,
    lead_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_lead(session, lead_id, lead, request_user)


@router.delete(
    "/{lead_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["delete_lead"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_lead(
    lead_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_lead(session, lead_id, request_user)


@router.get(
    "/upload/template/",
    dependencies=[
        Depends(authorize(permission_mapper["download_upload_lead_template"]))
    ],
)
async def download_upload_lead_template():
    lead_template = await get_version_upload_template(schemas.LeadValidator)

    buffer = io.BytesIO()
    with pd.ExcelWriter(buffer, engine="openpyxl") as writer:
        lead_template.to_excel(writer, sheet_name="Lead", index=False)
    buffer.seek(0)

    return StreamingResponse(
        buffer,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": 'attachment; filename="Lead Template.xlsx"'},
    )


@router.post(
    "/upload/",
    dependencies=[Depends(authorize(permission_mapper["upload_lead"]))],
    status_code=status.HTTP_201_CREATED,
)
async def upload_lead(
    lead_upload: Annotated[schemas.LeadUpload, Form(media_type="multipart/form-data")],
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.handle_lead_upload(session, lead_upload, request_user)


@router.get(
    "/download/",
    dependencies=[Depends(authorize(permission_mapper["download_lead"]))],
)
async def download_lead(
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    buffer = await utils.handle_lead_download(session, request_user)
    return StreamingResponse(
        buffer,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": 'attachment; filename="Leads.xlsx"'},
    )
