from fastapi import HTTPException, status
import io
import re

from be_core_services.contact.utils import create_contact
from be_core_services.contact.schemas import AddressCreate, ContactCreate, IndustryCreate
import pandas as pd
from pydantic import ValidationError
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_core_services.contact.repositories import ContactRepository
from . import enums, schemas, repositories


async def create_customer(
    db: AsyncSession,
    customer: schemas.CustomerCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.CustomerRepository(db)
    obj = await repo.acreate(
        **customer.model_dump(),
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
        _commit=True,
    )
    return obj


async def retrieve_customer(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.CustomerRepository(db, organization_id=group.organization_id)
    obj = await repo.aget_or_404(pk)
    return obj


async def list_customers(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.CustomerFilter,
    ordering: list[enums.CustomerOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.CustomerRepository(db, organization_id=group.organization_id)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def delete_customer(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.CustomerRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)

def format_phone(phone):
    if phone is None:
        return None
    phone_str = str(phone).strip()
    # Remove spaces and non-digit characters except '+'
    phone_str = re.sub(r"[^\d+]", "", phone_str)
    # Convert local format starting with '0' to international '+62'
    phone_str = "+62" + phone_str[1:]
    return phone_str

async def handle_customer_upload(
    db: AsyncSession, customer_upload: schemas.CustomerUpload, request_user: User
):
    buffer = await customer_upload.file.read()
    df = pd.read_excel(io.BytesIO(buffer), sheet_name="Customer")

    created_customers = []
    errors = []

    for idx, row in df.iterrows():
        try:
            # Prepare contact data (adjust fields as needed)
            contact = ContactCreate(
                name=row["name"],
                legal_type=row["legal_type"],
                tax_id=str(row["tax_id"]) if row.get("tax_id") is not None else None,
                industries=[IndustryCreate(industry_type=row["industry_type"])],
                addresses=[
                    AddressCreate(
                        address_type="Main",
                        name=row["contact_name"],
                        email=row.get("contact_email"),
                        address=row["address"],
                        phone=format_phone(row.get("phone_number")),
                    )
                ],
                tag_ids=[],
            )
            contact_obj = await create_contact(db, contact, request_user, _commit=False)

            # Create customer using contact pk
            customer_create = schemas.CustomerCreate(contact_id=contact_obj.pk, tag_ids=[])
            customer = await create_customer(db, customer_create, request_user)
            created_customers.append(customer)
        except Exception as e:
            errors.append({"row": idx + 2, "error": str(e)})
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail=f"Error on row {idx + 2}: {str(e)}"
            )

    await db.commit()
    return {"created": len(created_customers)}, 201


async def handle_customer_download(
    db: AsyncSession,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.CustomerRepository(db, organization_id=group.organization_id)
    customers = await repo.arecords()

    customer_rows = []
    for customer in customers:
        contact = customer.contact
        industries = ", ".join([ind.industry_type.value for ind in contact.industries]) if contact.industries else ""
        addresses = "; ".join([addr.address for addr in contact.addresses]) if contact.addresses else ""
        tags = ", ".join([tag.name for tag in contact.tags]) if contact.tags else ""
        phone = contact.addresses[0].phone if contact.addresses and hasattr(contact.addresses[0], "phone") else ""
        email = contact.addresses[0].email if contact.addresses and hasattr(contact.addresses[0], "email") else ""
        customer_rows.append(
            {
                "Name": contact.name,
                "Legal Type": contact.legal_type.value if contact.legal_type else None,
                "Tax ID": contact.tax_id,
                "Industries": industries,
                "Addresses": addresses,
                "Phone": phone,
                "Email": email,
                "Tags": tags,
            }
        )

    df = pd.DataFrame(customer_rows)
    output = io.BytesIO()
    with pd.ExcelWriter(output, engine="openpyxl") as writer:
        df.to_excel(writer, sheet_name="Customer", index=False)
    output.seek(0)
    return output
