import io
from typing import Annotated

from be_core_services.contact.utils import generate_vendor_customer_upload_template
import pandas as pd
from fastapi import APIRouter, Depends, Form, Query, status
from fastapi.responses import StreamingResponse
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_kit.exim.utils import get_version_upload_template
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user, authorize
from ..databases import get_async_session
from . import enums, schemas, utils
from .permissions import permission_mapper


router = APIRouter(tags=["customers"], prefix="/customer")


@router.post(
    "/",
    dependencies=[Depends(authorize(permission_mapper["create_customer"]))],
    response_model=schemas.Customer,
    status_code=status.HTTP_201_CREATED,
)
async def create_customer(
    customer: schemas.CustomerCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_customer(session, customer, request_user)


@router.get(
    "/{customer_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["retrieve_customer"]))],
    response_model=schemas.Customer,
)
async def retrieve_customer(
    customer_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_customer(session, customer_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(permission_mapper["list_customer"]))],
    response_model=schemas.PaginatedCustomer,
)
async def list_customer(
    pagination: PaginationQuery = Depends(),
    filters: schemas.CustomerFilter = Depends(),
    ordering: Annotated[list[enums.CustomerOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_customers(
        session, pagination, filters, ordering, request_user
    )


@router.get(
    "/options/",
    dependencies=[Depends(authorize(permission_mapper["list_customer_option"]))],
    response_model=schemas.PaginatedCustomerOpt,
)
async def list_customer_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.CustomerFilter = Depends(),
    ordering: Annotated[list[enums.CustomerOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_customers(
        session, pagination, filters, ordering, request_user
    )


@router.delete(
    "/{customer_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["delete_customer"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_customer(
    customer_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_customer(session, customer_id, request_user)


@router.get(
    "/upload/template/",
    dependencies=[
        Depends(authorize(permission_mapper["download_upload_customer_template"]))
    ],
)
async def download_upload_customer_template(
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    output, filename = await generate_vendor_customer_upload_template(
        schema_validator=schemas.CustomerValidator,
        sheet_name="Customer",
        filename="Customer Template.xlsx",
    )
    return StreamingResponse(
        output,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": f'attachment; filename="{filename}"'},
    )


@router.post(
    "/upload/",
    dependencies=[Depends(authorize(permission_mapper["upload_customer"]))],
    status_code=status.HTTP_201_CREATED,
)
async def upload_customer(
    customer_upload: Annotated[
        schemas.CustomerUpload, Form(media_type="multipart/form-data")
    ],
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.handle_customer_upload(session, customer_upload, request_user)


@router.get(
    "/download/",
    dependencies=[Depends(authorize(permission_mapper["download_customer"]))],
)
async def download_customer(
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    buffer = await utils.handle_customer_download(session, request_user)
    return StreamingResponse(
        buffer,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": 'attachment; filename="Customer.xlsx"'},
    )
