import io

import pandas as pd
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from . import enums, schemas, repositories


# Warehouse


async def create_warehouse(
    db: AsyncSession,
    warehouse: schemas.WarehouseCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.WarehouseRepository(db, organization_id=group.organization_id)
    obj = await repo.acreate(
        **warehouse.model_dump(),
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
    )
    return obj


async def retrieve_warehouse(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.WarehouseRepository(db, organization_id=group.organization_id)
    obj = await repo.aget_or_404(pk)
    return obj


async def update_warehouse(
    db: AsyncSession,
    pk: int,
    warehouse: schemas.WarehouseUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.WarehouseRepository(db, organization_id=group.organization_id)
    obj = await repo.aupdate(pk, **warehouse.model_dump())
    return obj


async def list_warehouses(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.WarehouseFilter,
    ordering: list[enums.WarehouseOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.WarehouseRepository(db, organization_id=group.organization_id)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def delete_warehouse(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.WarehouseRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)
