from typing import Annotated

from fastapi import APIRouter, Depends, Query, status
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user, authorize
from ..databases import get_async_session
from . import enums, schemas, utils
from .permissions import permission_mapper


router = APIRouter(tags=["warehouses"], prefix="/warehouse")


@router.post(
    "/",
    dependencies=[Depends(authorize(permission_mapper["create_warehouse"]))],
    response_model=schemas.Warehouse,
    status_code=status.HTTP_201_CREATED,
)
async def create_warehouse(
    warehouse: schemas.WarehouseCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_warehouse(session, warehouse, request_user)


@router.get(
    "/{warehouse_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["retrieve_warehouse"]))],
    response_model=schemas.Warehouse,
)
async def retrieve_warehouse(
    warehouse_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_warehouse(session, warehouse_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(permission_mapper["list_warehouse"]))],
    response_model=schemas.PaginatedWarehouse,
)
async def list_warehouse(
    pagination: PaginationQuery = Depends(),
    filters: schemas.WarehouseFilter = Depends(),
    ordering: Annotated[list[enums.WarehouseOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_warehouses(
        session, pagination, filters, ordering, request_user
    )


@router.get(
    "/options/",
    dependencies=[Depends(authorize(permission_mapper["list_warehouse_option"]))],
    response_model=schemas.PaginatedWarehouseOpt,
)
async def list_warehouse_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.WarehouseFilter = Depends(),
    ordering: Annotated[list[enums.WarehouseOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_warehouses(
        session, pagination, filters, ordering, request_user
    )


@router.put(
    "/{warehouse_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["update_warehouse"]))],
    response_model=schemas.Warehouse,
)
async def update_warehouse(
    warehouse_id: int,
    warehouse: schemas.WarehouseUpdate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_warehouse(session, warehouse_id, warehouse, request_user)


@router.delete(
    "/{warehouse_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["delete_warehouse"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_warehouse(
    warehouse_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_warehouse(session, warehouse_id, request_user)
