from fastapi import File, UploadFile
from typing_extensions import Annotated
from pydantic import PositiveInt, StringConstraints

from be_kit.paginations import BasePaginatedResponse
from be_kit.schemas import BaseSchema, BaseORMSchema, NominalType
from be_core_services.country.schemas import CurrencyMin
from be_uam.kit.schemas import MetadataMixin
from be_uam.user.schemas import UserMin
from be_accounting.coa.schemas import AccountMin
from be_accounting.journal.schemas import EntryMin
from ..warehouse.schemas import WarehouseMin
from . import enums


# ProductCategory


class BaseProductCategory:
    name: str
    description: str | None


class ProductCategoryCreate(BaseProductCategory, BaseSchema):
    pass


class ProductCategoryUpdate(BaseProductCategory, BaseSchema):
    pass


class ProductCategory(MetadataMixin, BaseProductCategory, BaseORMSchema):
    pass


class PaginatedProductCategory(BasePaginatedResponse):
    items: list[ProductCategory]


class ProductCategoryFilter(BaseSchema):
    name: str | None = None
    description: str | None = None


class ProductCategoryMin(BaseProductCategory, BaseORMSchema):
    pass


class ProductCategoryOpt(ProductCategoryMin):
    pass


class PaginatedProductCategoryOpt(BasePaginatedResponse):
    items: list[ProductCategoryOpt]


# Product


class BaseProduct:
    name: str
    description: str | None
    product_removal_strategy: enums.ProductRemovalStrategy
    product_type: enums.ProductType
    product_unit: enums.ProductUnit
    price: NominalType


class ProductIdentifierMixin:
    identifier: Annotated[
        str,
        StringConstraints(min_length=1, max_length=10),
    ]


class ProductCreate(BaseProduct, BaseSchema):
    category_id: PositiveInt
    currency_id: PositiveInt
    account_id: PositiveInt | None


class ProductUpdate(BaseProduct, BaseSchema):
    category_id: PositiveInt
    currency_id: PositiveInt
    account_id: PositiveInt | None


class Product(MetadataMixin, ProductIdentifierMixin, BaseProduct, BaseORMSchema):
    category: ProductCategoryMin
    currency: CurrencyMin
    account: AccountMin | None


class PaginatedProduct(BasePaginatedResponse):
    items: list[Product]


class ProductFilter(BaseSchema):
    name: str | None = None
    description: str | None = None
    identifier: str | None = None
    product_type: enums.ProductType | None = None
    product_unit: enums.ProductUnit | None = None
    category_id: PositiveInt | None = None
    currency_id: PositiveInt | None = None
    account_id: PositiveInt | None = None
    warehouse_id: PositiveInt | None = None


class ProductMin(BaseProduct, BaseORMSchema):
    category: ProductCategoryMin
    currency: CurrencyMin
    identifier: str


class ProductOpt(ProductMin):
    available_warehouses: list[WarehouseMin] = []
    pass


class PaginatedProductOpt(BasePaginatedResponse):
    items: list[ProductOpt]


# ProductStock


class BaseProductStock:
    available: NominalType


class ReservedStockMixin:
    reserved: NominalType
    locked: NominalType


class ProductStockCreate(BaseProductStock, BaseSchema):
    product_id: PositiveInt
    warehouse_id: PositiveInt


class ProductStockUpdate(BaseProductStock, BaseSchema):
    pass


class ProductStock(MetadataMixin, ReservedStockMixin, BaseProductStock, BaseORMSchema):
    product: ProductMin
    warehouse: WarehouseMin


class PaginatedProductStock(BasePaginatedResponse):
    items: list[ProductStock]


class ProductStockFilter(BaseSchema):
    available: NominalType | None = None
    reserved: NominalType | None = None
    locked: NominalType | None = None
    product_id: PositiveInt | None = None
    warehouse_id: PositiveInt | None = None


class ProductStockMin(BaseProductStock, BaseORMSchema):
    product: ProductCategoryMin
    warehouse: WarehouseMin


class ProductStockOpt(ProductStockMin):
    pass


class PaginatedProductStockOpt(BasePaginatedResponse):
    items: list[ProductStockOpt]


# ProductMovement


class BaseProductMovementItem:
    quantity: NominalType


class ProductMovementItemCreateUpdate(BaseProductMovementItem, BaseSchema):
    product_id: PositiveInt


class ProductMovementItem(BaseProductMovementItem, BaseORMSchema):
    product: ProductMin


class BaseProductMovement:
    movement_type: enums.ProductMovementType
    note: str | None


class ProductMovementIdentifierMixin:
    identifier: Annotated[
        str,
        StringConstraints(min_length=1, max_length=11),
    ]


class ProductMovementStatusMixin:
    status: enums.ProductMovementStatus


class ProductMovementCreate(BaseProductMovement, BaseSchema):
    source_warehouse_id: PositiveInt | None
    destination_warehouse_id: PositiveInt | None
    approver_id: PositiveInt | None
    items: list[ProductMovementItemCreateUpdate]


class ProductMovementUpdate(BaseProductMovement, BaseSchema):
    source_warehouse_id: PositiveInt | None
    destination_warehouse_id: PositiveInt | None
    approver_id: PositiveInt | None
    items: list[ProductMovementItemCreateUpdate]


class ProductMovementStatusUpdate(ProductMovementStatusMixin, BaseSchema):
    pass


class ProductMovement(
    MetadataMixin,
    ProductMovementStatusMixin,
    ProductMovementIdentifierMixin,
    BaseProductMovement,
    BaseORMSchema,
):
    source_warehouse: WarehouseMin | None
    destination_warehouse: WarehouseMin | None
    approver: UserMin | None
    journal_entry: EntryMin | None
    items: list[ProductMovementItem]


class ProductMovementDetail(ProductMovement):
    items: list[ProductMovementItem]


class PaginatedProductMovement(BasePaginatedResponse):
    items: list[ProductMovement]


class ProductMovementItemView(
    BaseProductMovementItem,
    MetadataMixin,
    ProductMovementStatusMixin,
    ProductMovementIdentifierMixin,
    BaseProductMovement,
    BaseORMSchema,
):
    source_warehouse: WarehouseMin | None
    destination_warehouse: WarehouseMin | None
    approver: UserMin | None
    product: ProductMin


class PaginatedProductMovementItemView(BasePaginatedResponse):
    items: list[ProductMovementItemView]


class ProductMovementFilter(BaseSchema):
    identifier: str | None = None
    status: enums.ProductMovementStatus | None = None
    movement_type: enums.ProductMovementType | None = None
    source_warehouse_id: PositiveInt | None = None
    destination_warehouse_id: PositiveInt | None = None
    approver_id: PositiveInt | None = None


class ProductMovementItemFilter(BaseSchema):
    identifier: str | None = None
    product_id: PositiveInt | None = None
    status: enums.ProductMovementStatus | None = None
    movement_type: enums.ProductMovementType | None = None
    source_warehouse_id: PositiveInt | None = None
    destination_warehouse_id: PositiveInt | None = None
    approver_id: PositiveInt | None = None


class ProductMovementMin(
    ProductMovementStatusMixin,
    ProductMovementIdentifierMixin,
    BaseProductMovement,
    BaseORMSchema,
):
    pass


class ProductMovementOpt(ProductMovementMin):
    pass


class PaginatedProductMovementOpt(BasePaginatedResponse):
    items: list[ProductMovementOpt]


class ProductUpload(BaseSchema):
    file: Annotated[UploadFile, File()]


class ProductValidator(BaseSchema):
    name: Annotated[str, StringConstraints(min_length=1, max_length=100)]
    category: Annotated[str, StringConstraints(min_length=1, max_length=100)]
    description: Annotated[str, StringConstraints(min_length=1, max_length=100)]
    product_type: enums.ProductType
    product_unit: enums.ProductUnit
    price: NominalType
    currency_iso_code: Annotated[str, StringConstraints(min_length=3, max_length=3)]


UserMin.model_rebuild()
