from datetime import date
from decimal import Decimal

from sqlalchemy import (
    BigInteger,
    Enum,
    ForeignKey,
    String,
    UniqueConstraint,
    Date,
)
from sqlalchemy.orm import Mapped, mapped_column, relationship

from be_kit.databases import BaseModel, SoftDeleteMixin, NominalField
from be_uam.user.models import User
from be_uam.kit.models import MetadataMixin, OrganizationMixin
from be_core_services.country.models import Currency
from be_accounting.coa.models import Account
from be_accounting.journal.models import Entry
from ..warehouse.models import Warehouse
from . import enums


class ProductCategory(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "inventory_product_category"
    __table_args__ = (UniqueConstraint("organization_id", "name"),)

    name: Mapped[str] = mapped_column(String(100), index=True)
    description: Mapped[str | None] = mapped_column(String(500), nullable=True)


class Product(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "inventory_product"
    __table_args__ = (UniqueConstraint("organization_id", "name"),)

    category_id: Mapped[int] = mapped_column(
        ForeignKey(f"{ProductCategory.__tablename__}.pk", ondelete="RESTRICT"),
    )
    category: Mapped[ProductCategory] = relationship(
        "ProductCategory",
        passive_deletes=True,
        lazy="selectin",
    )
    name: Mapped[str] = mapped_column(String(100), index=True)
    description: Mapped[str | None] = mapped_column(String(500), nullable=True)
    identifier: Mapped[str] = mapped_column(String(10), unique=True, index=True)
    product_removal_strategy: Mapped[enums.ProductRemovalStrategy] = mapped_column(
        Enum(enums.ProductRemovalStrategy),
        nullable=False,
    )
    product_type: Mapped[enums.ProductType] = mapped_column(
        Enum(enums.ProductType),
        nullable=False,
    )
    product_unit: Mapped[enums.ProductUnit] = mapped_column(
        Enum(enums.ProductUnit),
        nullable=False,
    )
    currency_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Currency.__tablename__}.pk", ondelete="RESTRICT"),
    )
    currency: Mapped[Currency] = relationship(
        "Currency",
        passive_deletes=True,
        lazy="selectin",
    )
    price: Mapped[Decimal] = mapped_column(NominalField, default=Decimal("0.00"))
    account_id: Mapped[int | None] = mapped_column(
        ForeignKey(f"{Account.__tablename__}.pk", ondelete="RESTRICT"),
        nullable=True,
        default=None,
    )
    account: Mapped[Account | None] = relationship(
        "Account",
        passive_deletes=True,
        lazy="selectin",
    )

    stocks: Mapped[list["ProductStock"]] = relationship(
        "ProductStock",
        back_populates="product",
        lazy="selectin",
    )


class ProductCounter(OrganizationMixin, BaseModel):
    __tablename__ = "inventory_product_counter"
    __table_args__ = (UniqueConstraint("organization_id", "counter_date"),)

    counter_date: Mapped[date] = mapped_column(Date, index=True)
    counter: Mapped[int] = mapped_column(BigInteger, default=0)


class ProductStock(OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "inventory_product_stock"
    __table_args__ = (
        UniqueConstraint("organization_id", "product_id", "warehouse_id"),
    )

    product_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Product.__tablename__}.pk", ondelete="RESTRICT"),
    )
    product: Mapped[Product] = relationship(
        "Product",
        passive_deletes=True,
        lazy="selectin",
    )
    warehouse_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Warehouse.__tablename__}.pk", ondelete="RESTRICT"),
    )
    warehouse: Mapped[Warehouse] = relationship(
        "Warehouse",
        passive_deletes=True,
        lazy="selectin",
    )
    available: Mapped[Decimal] = mapped_column(NominalField, default=Decimal("0.00"))
    reserved: Mapped[Decimal] = mapped_column(NominalField, default=Decimal("0.00"))
    locked: Mapped[Decimal] = mapped_column(NominalField, default=Decimal("0.00"))

    product: Mapped["Product"] = relationship(
        "Product",
        back_populates="stocks",
        passive_deletes=True,
        lazy="selectin",
    )


class ProductMovement(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "inventory_product_movement"

    source_warehouse_id: Mapped[int | None] = mapped_column(
        ForeignKey(f"{Warehouse.__tablename__}.pk", ondelete="RESTRICT"),
        nullable=True,
    )
    source_warehouse: Mapped[Warehouse | None] = relationship(
        "Warehouse",
        foreign_keys=[source_warehouse_id],
        passive_deletes=True,
        lazy="selectin",
    )
    destination_warehouse_id: Mapped[int | None] = mapped_column(
        ForeignKey(f"{Warehouse.__tablename__}.pk", ondelete="RESTRICT"),
        nullable=True,
    )
    destination_warehouse: Mapped[Warehouse | None] = relationship(
        "Warehouse",
        foreign_keys=[destination_warehouse_id],
        passive_deletes=True,
        lazy="selectin",
    )
    identifier: Mapped[str] = mapped_column(String(11), unique=True, index=True)
    movement_type: Mapped[enums.ProductMovementType] = mapped_column(
        Enum(enums.ProductMovementType),
        nullable=False,
    )
    status: Mapped[enums.ProductMovementStatus] = mapped_column(
        Enum(enums.ProductMovementStatus),
        nullable=False,
        default=enums.ProductMovementStatus.DRAFT,
    )
    items: Mapped[list["ProductMovementItem"]] = relationship(
        "ProductMovementItem",
        back_populates="movement",
        cascade="all, delete-orphan",
        lazy="selectin",
    )
    approver_id: Mapped[int | None] = mapped_column(
        ForeignKey("uam_user.pk", ondelete="SET NULL"),
        nullable=True,
    )
    approver: Mapped[User | None] = relationship(
        "User",
        foreign_keys=[approver_id],
        passive_deletes=True,
        lazy="selectin",
    )
    note: Mapped[str | None] = mapped_column(String(500), nullable=True)

    journal_entry_id: Mapped[int | None] = mapped_column(
        BigInteger, ForeignKey(Entry.pk, ondelete="SET NULL"), nullable=True
    )
    journal_entry: Mapped["Entry | None"] = relationship(
        Entry,
        foreign_keys=[journal_entry_id],
        lazy="selectin",
    )

class ProductMovementItem(BaseModel):
    __tablename__ = "inventory_product_movement_item"
    __table_args__ = (UniqueConstraint("movement_id", "product_id"),)

    movement_id: Mapped[int] = mapped_column(
        ForeignKey(f"{ProductMovement.__tablename__}.pk", ondelete="CASCADE"),
    )
    movement: Mapped[ProductMovement] = relationship(
        "ProductMovement",
        passive_deletes=True,
        lazy="selectin",
    )
    product_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Product.__tablename__}.pk", ondelete="RESTRICT"),
    )
    product: Mapped[Product] = relationship(
        "Product",
        passive_deletes=True,
        lazy="selectin",
    )
    quantity: Mapped[Decimal] = mapped_column(NominalField, nullable=False)


class ProductMovementCounter(OrganizationMixin, BaseModel):
    __tablename__ = "inventory_product_movement_counter"
    __table_args__ = (UniqueConstraint("organization_id", "counter_date"),)

    counter_date: Mapped[date] = mapped_column(Date, index=True)
    counter: Mapped[int] = mapped_column(BigInteger, default=0)
