import io
import pandas as pd
from typing import Annotated

from openpyxl import load_workbook
from openpyxl.worksheet.datavalidation import DataValidation

from be_kit.exim.utils import get_version_upload_template
from fastapi import APIRouter, Depends, Query, status
from fastapi.params import Form
from fastapi.responses import StreamingResponse
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user, authorize
from ..databases import get_async_session
from . import enums, schemas, utils
from .permissions import (
    product_category_permission_mapper,
    product_permission_mapper,
    product_stock_permission_mapper,
    product_movement_permission_mapper,
)


# ProductStock


product_category_router = APIRouter(
    tags=["product_categories"], prefix="/product_category"
)


@product_category_router.post(
    "/",
    dependencies=[
        Depends(
            authorize(product_category_permission_mapper["create_product_category"])
        )
    ],
    response_model=schemas.ProductCategory,
    status_code=status.HTTP_201_CREATED,
)
async def create_product_category(
    product_category: schemas.ProductCategoryCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_product_category(session, product_category, request_user)


@product_category_router.get(
    "/{product_category_id:int}/",
    dependencies=[
        Depends(
            authorize(product_category_permission_mapper["retrieve_product_category"])
        )
    ],
    response_model=schemas.ProductCategory,
)
async def retrieve_product_category(
    product_category_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_product_category(
        session, product_category_id, request_user
    )


@product_category_router.get(
    "/",
    dependencies=[
        Depends(authorize(product_category_permission_mapper["list_product_category"]))
    ],
    response_model=schemas.PaginatedProductCategory,
)
async def list_product_category(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ProductCategoryFilter = Depends(),
    ordering: Annotated[list[enums.ProductCategoryOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_product_categories(
        session, pagination, filters, ordering, request_user
    )


@product_category_router.get(
    "/options/",
    dependencies=[
        Depends(
            authorize(
                product_category_permission_mapper["list_product_category_option"]
            )
        )
    ],
    response_model=schemas.PaginatedProductCategoryOpt,
)
async def list_product_category_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ProductCategoryFilter = Depends(),
    ordering: Annotated[list[enums.ProductCategoryOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_product_categories(
        session, pagination, filters, ordering, request_user
    )


@product_category_router.put(
    "/{product_category_id:int}/",
    dependencies=[
        Depends(
            authorize(product_category_permission_mapper["update_product_category"])
        )
    ],
    response_model=schemas.ProductCategory,
)
async def update_product_category(
    product_category_id: int,
    product_category: schemas.ProductCategoryUpdate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_product_category(
        session, product_category_id, product_category, request_user
    )


@product_category_router.delete(
    "/{product_category_id:int}/",
    dependencies=[
        Depends(
            authorize(product_category_permission_mapper["delete_product_category"])
        )
    ],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_product_category(
    product_category_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_product_category(session, product_category_id, request_user)


# Product


product_router = APIRouter(tags=["products"], prefix="/product")


@product_router.post(
    "/",
    dependencies=[Depends(authorize(product_permission_mapper["create_product"]))],
    response_model=schemas.Product,
    status_code=status.HTTP_201_CREATED,
)
async def create_product(
    product: schemas.ProductCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_product(session, product, request_user)


@product_router.get(
    "/{product_id:int}/",
    dependencies=[Depends(authorize(product_permission_mapper["retrieve_product"]))],
    response_model=schemas.Product,
)
async def retrieve_product(
    product_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_product(session, product_id, request_user)


@product_router.get(
    "/",
    dependencies=[Depends(authorize(product_permission_mapper["list_product"]))],
    response_model=schemas.PaginatedProduct,
)
async def list_product(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ProductFilter = Depends(),
    ordering: Annotated[list[enums.ProductOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_products(
        session, pagination, filters, ordering, request_user
    )


@product_router.get(
    "/options/",
    dependencies=[Depends(authorize(product_permission_mapper["list_product_option"]))],
    response_model=schemas.PaginatedProductOpt,
)
async def list_product_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ProductFilter = Depends(),
    ordering: Annotated[list[enums.ProductOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_products(
        session, pagination, filters, ordering, request_user
    )


@product_router.put(
    "/{product_id:int}/",
    dependencies=[Depends(authorize(product_permission_mapper["update_product"]))],
    response_model=schemas.Product,
)
async def update_product(
    product_id: int,
    product: schemas.ProductUpdate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_product(session, product_id, product, request_user)


@product_router.delete(
    "/{product_id:int}/",
    dependencies=[Depends(authorize(product_permission_mapper["delete_product"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_product(
    product_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_product(session, product_id, request_user)

@product_router.get(
    "/upload/template/",
    dependencies=[
        Depends(authorize(product_permission_mapper["download_upload_product_template"]))
    ],
)
async def download_upload_product_template(
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session)
):
    product_template = await get_version_upload_template(schemas.ProductValidator)

    # Fetch categories and currencies from DB
    categories = await utils.get_all_product_category_names(session, request_user)
    currencies = await utils.get_all_currency_iso_codes(session, request_user)

    buffer = io.BytesIO()
    with pd.ExcelWriter(buffer, engine="openpyxl") as writer:
        product_template.to_excel(writer, sheet_name="Product", index=False)
    buffer.seek(0)

    wb = load_workbook(buffer)
    ws = wb["Product"]

    # Add a hidden sheet for dropdown sources
    ws_lists = wb.create_sheet("lists")
    for i, cat in enumerate(categories, 1):
        ws_lists[f"A{i}"] = cat
    for i, iso in enumerate(currencies, 1):
        ws_lists[f"B{i}"] = iso
    wb.active = 0  # Set back to main sheet

    # Import your enums
    from .enums import ProductType, ProductUnit

    # Get enum values
    product_type_values = [e.value for e in ProductType]
    product_unit_values = [e.value for e in ProductUnit]

    # Find columns
    header = [cell.value for cell in ws[1]]
    from openpyxl.utils import get_column_letter
    category_col = get_column_letter(header.index("category") + 1)
    currency_col = get_column_letter(header.index("currency_iso_code") + 1)
    product_type_col = get_column_letter(header.index("product_type") + 1)
    product_unit_col = get_column_letter(header.index("product_unit") + 1)

    # Add data validation (dropdown) for product_type
    dv_type = DataValidation(type="list", formula1=f'"{",".join(product_type_values)}"', allow_blank=True)
    ws.add_data_validation(dv_type)
    dv_type.add(f"{product_type_col}2:{product_type_col}1048576")

    # Add data validation (dropdown) for product_unit
    dv_unit = DataValidation(type="list", formula1=f'"{",".join(product_unit_values)}"', allow_blank=True)
    ws.add_data_validation(dv_unit)
    dv_unit.add(f"{product_unit_col}2:{product_unit_col}1048576")

    # Add data validation (dropdown) for category
    dv_cat = DataValidation(type="list", formula1=f'=lists!$A$1:$A${len(categories)}', allow_blank=True)
    ws.add_data_validation(dv_cat)
    dv_cat.add(f"{category_col}2:{category_col}1048576")

    # Add data validation (dropdown) for currency iso code
    dv_cur = DataValidation(type="list", formula1=f'=lists!$B$1:$B${len(currencies)}', allow_blank=True)
    ws.add_data_validation(dv_cur)
    dv_cur.add(f"{currency_col}2:{currency_col}1048576")

    # Optionally hide the lists sheet
    ws_lists.sheet_state = "hidden"

    output = io.BytesIO()
    wb.save(output)
    output.seek(0)

    return StreamingResponse(
        output,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": 'attachment; filename="Product Template.xlsx"'},
    )


@product_router.post(
    "/upload/",
    dependencies=[Depends(authorize(product_permission_mapper["upload_product"]))],
    status_code=status.HTTP_201_CREATED,
)
async def upload_product(
    product_upload: Annotated[
        schemas.ProductUpload, Form(media_type="multipart/form-data")
    ],
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.handle_product_upload(session, product_upload, request_user)


@product_router.get(
    "/download/",
    dependencies=[Depends(authorize(product_permission_mapper["download_product"]))],
)
async def download_product(
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    buffer = await utils.handle_product_download(session, request_user)
    return StreamingResponse(
        buffer,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": 'attachment; filename="Products.xlsx"'},
    )

# ProductStock


product_stock_router = APIRouter(tags=["product_stocks"], prefix="/product_stock")


@product_stock_router.post(
    "/",
    dependencies=[
        Depends(authorize(product_stock_permission_mapper["create_product_stock"]))
    ],
    response_model=schemas.ProductStock,
    status_code=status.HTTP_201_CREATED,
)
async def create_product_stock(
    product_stock: schemas.ProductStockCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_product_stock(session, product_stock, request_user)


@product_stock_router.get(
    "/{product_stock_id:int}/",
    dependencies=[
        Depends(authorize(product_stock_permission_mapper["retrieve_product_stock"]))
    ],
    response_model=schemas.ProductStock,
)
async def retrieve_product_stock(
    product_stock_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_product_stock(session, product_stock_id, request_user)


@product_stock_router.get(
    "/",
    dependencies=[
        Depends(authorize(product_stock_permission_mapper["list_product_stock"]))
    ],
    response_model=schemas.PaginatedProductStock,
)
async def list_product_stock(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ProductStockFilter = Depends(),
    ordering: Annotated[list[enums.ProductStockOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_product_stocks(
        session, pagination, filters, ordering, request_user
    )


@product_stock_router.get(
    "/options/",
    dependencies=[
        Depends(authorize(product_stock_permission_mapper["list_product_stock_option"]))
    ],
    response_model=schemas.PaginatedProductStockOpt,
)
async def list_product_stock_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ProductStockFilter = Depends(),
    ordering: Annotated[list[enums.ProductStockOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_product_stocks(
        session, pagination, filters, ordering, request_user
    )


@product_stock_router.put(
    "/{product_stock_id:int}/",
    dependencies=[
        Depends(authorize(product_stock_permission_mapper["update_product_stock"]))
    ],
    response_model=schemas.ProductStock,
)
async def update_product_stock(
    product_stock_id: int,
    product_stock: schemas.ProductStockUpdate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_product_stock(
        session, product_stock_id, product_stock, request_user
    )


@product_stock_router.delete(
    "/{product_stock_id:int}/",
    dependencies=[
        Depends(authorize(product_stock_permission_mapper["delete_product_stock"]))
    ],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_product_stock(
    product_stock_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_product_stock(session, product_stock_id, request_user)


# ProductMovement


product_movement_router = APIRouter(
    tags=["product_movements"], prefix="/product_movement"
)


@product_movement_router.post(
    "/",
    dependencies=[
        Depends(
            authorize(product_movement_permission_mapper["create_product_movement"])
        )
    ],
    response_model=schemas.ProductMovement,
    status_code=status.HTTP_201_CREATED,
)
async def create_product_movement(
    product_movement: schemas.ProductMovementCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_product_movement(session, product_movement, request_user)


@product_movement_router.get(
    "/{product_movement_id:int}/",
    dependencies=[
        Depends(
            authorize(product_movement_permission_mapper["retrieve_product_movement"])
        )
    ],
    response_model=schemas.ProductMovement,
)
async def retrieve_product_movement(
    product_movement_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_product_movement(
        session, product_movement_id, request_user
    )


@product_movement_router.get(
    "/",
    dependencies=[
        Depends(authorize(product_movement_permission_mapper["list_product_movement"]))
    ],
    response_model=schemas.PaginatedProductMovement,
)
async def list_product_movement(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ProductMovementFilter = Depends(),
    ordering: Annotated[list[enums.ProductMovementOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_product_movements(
        session, pagination, filters, ordering, request_user
    )

@product_movement_router.get(
    "/item/",
    dependencies=[
        Depends(authorize(product_movement_permission_mapper["list_product_movement"]))
    ],
    response_model=schemas.PaginatedProductMovementItemView,
)
async def list_product_movement_item(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ProductMovementItemFilter = Depends(),
    ordering: Annotated[list[enums.ProductMovementOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_product_movement_items(session, pagination, filters, ordering, request_user)

@product_movement_router.get(
    "/options/",
    dependencies=[
        Depends(
            authorize(
                product_movement_permission_mapper["list_product_movement_option"]
            )
        )
    ],
    response_model=schemas.PaginatedProductMovementOpt,
)
async def list_product_movement_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ProductMovementFilter = Depends(),
    ordering: Annotated[list[enums.ProductMovementOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_product_movements(
        session, pagination, filters, ordering, request_user
    )


@product_movement_router.put(
    "/{product_movement_id:int}/",
    dependencies=[
        Depends(
            authorize(product_movement_permission_mapper["update_product_movement"])
        )
    ],
    response_model=schemas.ProductMovement,
)
async def update_product_movement(
    product_movement_id: int,
    product_movement: schemas.ProductMovementUpdate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_product_movement(
        session, product_movement_id, product_movement, request_user
    )


@product_movement_router.put(
    "/{product_movement_id:int}/status/",
    dependencies=[
        Depends(
            authorize(
                product_movement_permission_mapper["update_product_movement_status"]
            )
        )
    ],
    response_model=schemas.ProductMovement,
)
async def update_product_movement_status(
    product_movement_id: int,
    status_update: schemas.ProductMovementStatusUpdate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_product_movement_status(
        session, product_movement_id, status_update, request_user
    )


@product_movement_router.put(
    "/{product_movement_id:int}/approve/",
    dependencies=[
        Depends(
            authorize(product_movement_permission_mapper["approve_product_movement"])
        )
    ],
    response_model=schemas.ProductMovement,
)
async def approve_product_movement(
    product_movement_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.approve_product_movement(
        session, product_movement_id, request_user
    )


@product_movement_router.delete(
    "/{product_movement_id:int}/",
    dependencies=[
        Depends(
            authorize(product_movement_permission_mapper["delete_product_movement"])
        )
    ],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_product_movement(
    product_movement_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_product_movement(session, product_movement_id, request_user)

