import io
import re
from fastapi import HTTPException, status
import pandas as pd
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_core_services.contact.utils import create_contact
from be_core_services.contact.schemas import ContactCreate, IndustryCreate, AddressCreate

from . import schemas, repositories


async def create_vendor(
    db: AsyncSession,
    vendor: schemas.VendorCreate,
    request_user: User,
):
    vendor_data = vendor.model_dump()
    group = await request_user.awaitable_attrs.group
    repo = repositories.VendorRepository(
        db,
    )
    obj = await repo.acreate(
        **vendor_data,
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
    )
    return obj


async def retrieve_vendor(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.VendorRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    return obj


async def list_vendors(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.VendorFilter,
    request_user: User,
    as_options: bool = False,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.VendorRepository(
        db,
        organization_id=group.organization_id,
    )
    objs = await repo.arecords(filters, pagination, False)
    return objs


async def delete_vendor(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.VendorRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)


async def handle_vendor_download(
    db: AsyncSession,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.VendorRepository(db, organization_id=group.organization_id)
    vendors = await repo.arecords()

    vendor_rows = []
    for vendor in vendors:
        contact = vendor.contact
        industries = ", ".join([ind.industry_type.value for ind in contact.industries]) if contact.industries else ""
        addresses = "; ".join([addr.address for addr in contact.addresses]) if contact.addresses else ""
        tags = ", ".join([tag.name for tag in contact.tags]) if contact.tags else ""
        phone = contact.addresses[0].phone if contact.addresses and hasattr(contact.addresses[0], "phone") else ""
        email = contact.addresses[0].email if contact.addresses and hasattr(contact.addresses[0], "email") else ""
        vendor_rows.append(
            {
                "Name": contact.name,
                "Legal Type": contact.legal_type.value if contact.legal_type else None,
                "Tax ID": contact.tax_id,
                "Industries": industries,
                "Addresses": addresses,
                "Phone": phone,
                "Email": email,
                "Tags": tags,
            }
        )

    df = pd.DataFrame(vendor_rows)
    output = io.BytesIO()
    with pd.ExcelWriter(output, engine="openpyxl") as writer:
        df.to_excel(writer, sheet_name="vendor", index=False)
    output.seek(0)
    return output

def format_phone(phone):
    if phone is None:
        return None
    phone_str = str(phone).strip()
    # Remove spaces and non-digit characters except '+'
    phone_str = re.sub(r"[^\d+]", "", phone_str)
    # Convert local format starting with '0' to international '+62'
    phone_str = "+62" + phone_str[1:]
    return phone_str

async def handle_vendor_upload(
    db: AsyncSession,
    vendor_upload: schemas.VendorUpload,
    request_user: User,
):
    buffer = await vendor_upload.file.read()
    df = pd.read_excel(io.BytesIO(buffer), sheet_name="Vendor")

    created_vendors = []
    errors = []

    for idx, row in df.iterrows():
        try:
            # Prepare contact data (adjust fields as needed)
            contact = ContactCreate(
                name=row["name"],
                legal_type=row["legal_type"],
                tax_id=str(row["tax_id"]) if row.get("tax_id") is not None else None,
                industries=[IndustryCreate(industry_type=row["industry_type"])],
                addresses=[
                    AddressCreate(
                        address_type="Main",
                        name=row["contact_name"],
                        email=row.get("contact_email"),
                        address=row["address"],
                        phone=format_phone(row.get("phone_number")),
                    )
                ],
                tag_ids=[],
            )
            contact_obj = await create_contact(db, contact, request_user, _commit=False)

            # Create vendor using contact pk
            vendor_create = schemas.VendorCreate(contact_id=contact_obj.pk)
            vendor = await create_vendor(db, vendor_create, request_user)
            created_vendors.append(vendor)
        except Exception as e:
            errors.append({"row": idx + 2, "error": str(e)})
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail=f"Error on row {idx + 2}: {str(e)}"
            )

    await db.commit()
    return {"created": len(created_vendors), "errors": errors}, 201
