import io
from typing import Annotated
from be_kit.exim.utils import get_version_upload_template
from fastapi import APIRouter, Depends, Form, Query, status
from fastapi.responses import StreamingResponse
import pandas as pd
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.auth.utils import get_request_user, authorize
from be_uam.user.models import User
from be_core_services.contact.utils import generate_vendor_customer_upload_template
from . import schemas, utils, enums
from .permissions import vendor_permission_mapper
from ..databases import get_async_session
from ..settings import Settings


router = APIRouter(tags=["vendors"], prefix="/vendor")


@router.post(
    "/",
    dependencies=[Depends(authorize(vendor_permission_mapper["create_vendor"]))],
    response_model=schemas.Vendor,
)
async def create_vendor(
    vendor: schemas.VendorCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_vendor(session, vendor, request_user)


@router.get(
    "/{vendor_id:int}/",
    dependencies=[Depends(authorize(vendor_permission_mapper["retrieve_vendor"]))],
    response_model=schemas.Vendor,
)
async def retrieve_vendor(
    vendor_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_vendor(session, vendor_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(vendor_permission_mapper["list_vendor"]))],
    response_model=schemas.PaginatedVendor,
)
async def list_vendor(
    pagination: PaginationQuery = Depends(),
    filters: schemas.VendorFilter = Depends(),
    ordering: Annotated[list[enums.VendorOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_vendors(session, pagination, filters, request_user, ordering)


@router.get(
    "/options/",
    dependencies=[Depends(authorize(vendor_permission_mapper["list_vendor_option"]))],
    response_model=schemas.PaginatedVendorOpt,
)
async def list_vendor_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.VendorFilter = Depends(),
    ordering: Annotated[list[enums.VendorOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_vendors(
        session, pagination, filters, request_user, ordering
    )


@router.delete(
    "/{vendor_id:int}/",
    dependencies=[Depends(authorize(vendor_permission_mapper["delete_vendor"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_vendor(
    vendor_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_vendor(session, vendor_id, request_user)


@router.get(
    "/upload/template/",
    dependencies=[
        Depends(authorize(vendor_permission_mapper["download_upload_vendor_template"]))
    ],
)
async def download_upload_vendor_template(
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    output, filename = await generate_vendor_customer_upload_template(
        schema_validator=schemas.VendorValidator,
        sheet_name="Vendor",
        filename="Vendor Template.xlsx",
    )
    return StreamingResponse(
        output,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": f'attachment; filename="{filename}"'},
    )


@router.post(
    "/upload/",
    dependencies=[Depends(authorize(vendor_permission_mapper["upload_vendor"]))],
    status_code=status.HTTP_201_CREATED,
)
async def upload_vendor(
    vendor_upload: Annotated[
        schemas.VendorUpload, Form(media_type="multipart/form-data")
    ],
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.handle_vendor_upload(session, vendor_upload, request_user)


@router.get(
    "/download/",
    dependencies=[Depends(authorize(vendor_permission_mapper["download_vendor"]))],
)
async def download_vendor(
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    buffer = await utils.handle_vendor_download(session, request_user)
    return StreamingResponse(
        buffer,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": 'attachment; filename="Vendor.xlsx"'},
    )
