from decimal import Decimal
from be_core_services.country.models import VAT, Currency
from sqlalchemy import BigInteger, Enum, Numeric, ForeignKey, Integer, String
from sqlalchemy.ext.declarative import declared_attr
from sqlalchemy.orm import Mapped, mapped_column, relationship

from be_uam.user.models import User

from be_inventory.product.models import Product
from be_inventory.warehouse.models import Warehouse
from be_core_services.country.models import VAT
from .enums import TransactionStatus, TransactionPaymentTerm, TransactionDeliveryTerm

class CalculationMixin:
    untaxed_subtotal: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)
    global_discount: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)
    tax_value: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)
    discount_per_item: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)
    grand_total: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)
    global_dp: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)
    total_after_tax_and_discount: Mapped[Decimal] = mapped_column(Numeric(20, 4), nullable=True)

class TransactionMixin:
    status: Mapped["TransactionStatus"] = mapped_column(
        Enum(TransactionStatus), index=True, default=TransactionStatus.DRAFT
    )
    payment_term: Mapped["TransactionPaymentTerm"] = mapped_column(
        Enum(TransactionPaymentTerm), index=True
    )
    payment_n: Mapped["int | None"] = mapped_column(Integer, nullable=True)
    payment_dp_rate: Mapped["Decimal | None"] = mapped_column(
        Numeric(5, 4), nullable=True
    )
    payment_dp: Mapped["Decimal | None"] = mapped_column(Numeric(20, 4), nullable=True)
    delivery_term: Mapped["TransactionDeliveryTerm | None"] = mapped_column(
        Enum(TransactionDeliveryTerm), index=True, nullable=True
    )
    discount_rate: Mapped["Decimal | None"] = mapped_column(
        Numeric(5, 4), nullable=True
    )
    discount_amount: Mapped[Decimal] = mapped_column(Numeric(20, 4), nullable=True)
    total: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_discount: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_global_discount: Mapped[Decimal] = mapped_column(
        Numeric(20, 4), nullable=True
    )
    total_before_vat: Mapped[Decimal] = mapped_column(Numeric(20, 4), nullable=True)
    total_vat: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_net: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_net_after_dp: Mapped[Decimal] = mapped_column(Numeric(20, 4), nullable=True)
    total_after_tax_and_discount: Mapped[Decimal] = mapped_column(Numeric(20, 4), nullable=True)
    notes: Mapped[str] = mapped_column(String(255), nullable=True)

    untaxed_subtotal: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)
    global_discount: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)
    tax_value: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)
    discount_per_item: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)
    total: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)
    grand_total: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)
    global_dp: Mapped[Decimal | None] = mapped_column(Numeric(20, 4), nullable=True)

    @declared_attr
    @classmethod
    def currency_id(cls) -> Mapped[int]:
        return mapped_column(
            BigInteger,
            ForeignKey("core_services_currency.pk", ondelete="RESTRICT"),
        )

    @declared_attr
    @classmethod
    def currency(cls) -> Mapped["Currency"]:
        return relationship(
            "be_core_services.country.models.Currency",
            foreign_keys=[cls.currency_id],
        )

    @declared_attr
    @classmethod
    def approved_by_id(cls) -> Mapped[int | None]:
        return mapped_column(
            BigInteger,
            ForeignKey("uam_user.pk", ondelete="SET NULL"),
            default=None,
            nullable=True,
        )

    @declared_attr
    @classmethod
    def approved_by(cls) -> Mapped[User | None]:
        return relationship(User, foreign_keys=[cls.approved_by_id])


class TransactionItemMixin:
    quantity: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    price: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    discount_rate: Mapped["Decimal | None"] = mapped_column(
        Numeric(5, 4), nullable=True
    )
    discount_amount: Mapped["Decimal | None"] = mapped_column(
        Numeric(20, 4), nullable=True
    )
    vat_rate: Mapped[Decimal] = mapped_column(Numeric(5, 4))
    total: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_discount: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_global_discount: Mapped[Decimal] = mapped_column(
        Numeric(20, 4), nullable=True
    )
    total_before_vat: Mapped[Decimal] = mapped_column(Numeric(20, 4), nullable=True)
    total_vat: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_net: Mapped[Decimal] = mapped_column(Numeric(20, 4))

    @declared_attr
    def warehouse_id(cls) -> Mapped[int]:
        return mapped_column(
            BigInteger,
            ForeignKey(Warehouse.pk, ondelete="RESTRICT"),
        )

    @declared_attr
    def warehouse(cls) -> Mapped["Warehouse"]:
        return relationship(
            "be_inventory.warehouse.models.Warehouse", foreign_keys=[cls.warehouse_id], lazy="selectin"
        )

    @declared_attr
    def vat_id(cls) -> Mapped["int | None"]:
        return mapped_column(
            BigInteger,
            ForeignKey(VAT.pk, ondelete="RESTRICT"),
            nullable=True,
        )

    @declared_attr
    def vat(cls) -> Mapped["VAT | None"]:
        return relationship(
            "be_core_services.country.models.VAT", foreign_keys=[cls.vat_id], lazy="selectin"
        )

    @declared_attr
    def product_id(cls) -> Mapped[int]:
        return mapped_column(
            BigInteger,
            ForeignKey(Product.pk, ondelete="RESTRICT"),
        )

    @declared_attr
    def product(cls) -> Mapped["Product"]:
        return relationship(
            "be_inventory.product.models.Product", foreign_keys=[cls.product_id], lazy="selectin"
        )
